
/* Copyright (C) 2001-2008 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* Copyright: 1987-1990 by Apple Computer, Inc., all rights reserved. */

/* fs_glue.c */


/**********************************************************************************/

#include "fs_itype.h"
#include "fs_fontscal.h"
#include "fs_rtgah.h"

#define SHORTMUL(a,b) (FS_LONG)((FS_LONG)a * (FS_LONG)b)

#define FROUND( x, n, d, s ) \
        x = SHORTMUL(x, n); x += d; x >>= s;


#define SROUND( x, n, d, halfd ) \
    if ( x < 0 ) \
        { x = -x; x = SHORTMUL(x, n); x += halfd; x /= d; x = -x; } \
    else \
        { x = SHORTMUL(x, n); x += halfd; x /= d; }

static F26DOT6 fnt_FRound(scale_rec *rec, F26DOT6 value)
{
    FS_LONG N = rec->N;
    FS_LONG Dover2 = rec->D >> 1;
    FROUND(value, N, Dover2, rec->shift);
    return value;
}
static F26DOT6 fnt_SRound(scale_rec *rec, F26DOT6 value)
{
    FS_LONG N = rec->N;
    FS_LONG D = rec->D;
    FS_LONG Dover2 = D >> 1;
    SROUND(value, N, D, Dover2);
    return value;
}
static F26DOT6 fnt_FixRound(scale_rec *rec, F26DOT6 value)
{
    return FixMul(rec->scl, value);
}


F26DOT6 ScaleFuncCall(ScaleFunc f, scale_rec*r, F26DOT6 v)
{
    switch (f)
    {
    case SFI_fnt_FRound:
        return fnt_FRound(r, v);
    case SFI_fnt_SRound:
        return fnt_SRound(r, v);
    case SFI_fnt_FixRound:
        return fnt_FixRound(r, v);
    }
    return(0);
}

#ifdef FS_RENDER

#ifdef EDIT_MODE
FS_BYTE *instructions;
FS_LONG instructionLength;
FS_BYTE *preProgram = NULL;
FS_LONG preProgramLength = 0;
#endif

#ifdef NC_DEBUG
/**********************************************************************************/
static int indent;
FS_LONG nested;

static FS_VOID indent_fn()
{
    int i;

    FS_PRINTF(("%%"));
    for (i = 0; i <= indent; i++)
        FS_PRINTF((" "));
}
/**********************************************************************************/
static FS_VOID dump_matrix(FILECHAR *s, transMatrix mat)
{
    double z = 65536.0;
    FS_FIXED *m = &mat.transform[0][0];

    indent_fn();
    FS_PRINTF(("%s\n", s));
    indent_fn();
    FS_PRINTF(("%7.3f %7.3f %7.3f\n", m[0] / z, m[1] / z, m[2] / z));
    indent_fn();
    FS_PRINTF(("%7.3f %7.3f %7.3f\n", m[3] / z, m[4] / z, m[5] / z));
    indent_fn();
    FS_PRINTF(("%7.3f %7.3f %7.3f\n", m[6] / z, m[7] / z, m[8] / z));
}
/**********************************************************************************/
static dumpCVT(fsg_SplineKey *key)
{
    int i;
    int n = key->cvtCount;
    F26DOT6* cvt = (F26DOT6*)((FS_BYTE *)(key->memoryBases[PRIVATE_FONT_SPACE_BASE]) + key->offset_controlValues);

    FS_PRINTF(("scaled CVT\n"));
    for (i = 0; i < n; i++)
        FS_PRINTF(("%3d: %7.3f\n", i, cvt[i] / 64.0));
}
/**********************************************************************************/
FS_VOID dump_elementPtr(fnt_ElementType *p, FILECHAR *s)
{
    int i, j, nc, np;

    nc = p->nc;
    np = nc ? 1 + p->ep[nc - 1] - p->sp[0] : 0;

    indent_fn();
    FS_PRINTF(("dump_elementPtr -- %s\n", s));
    indent_fn();
    FS_PRINTF(("p=%p nc=%d np=%d\n", p, nc, np));

    indent_fn();
    FS_PRINTF(("sp[%p]=", p->sp));
    for (i = 0; i < nc; i++)
        FS_PRINTF(("%d ", p->sp[i]));
    FS_PRINTF(("\n"));

    indent_fn();
    FS_PRINTF(("ep[%p]=", p->ep));
    for (i = 0; i < nc; i++)
        FS_PRINTF(("%d ", p->ep[i]));
    FS_PRINTF(("\n"));

    indent_fn();
    FS_PRINTF(("-d- -f- -on- ----x-- ----y-- ---ox-- ---oy-- --oox-- --ooy--\n"));

    for (i = 0; i < nc; i++)
    {
        for (j = p->sp[i]; j <= p->ep[i]; j++)
        {
            indent_fn();
            FS_PRINTF(("%3d %02x  %3d %7.2f %7.2f %7.2f %7.2f %7d %7d\n", j,
                       p->f[j], p->onCurve[j], p->x[j] / 64.0, p->y[j] / 64.0,
                       p->ox[j] / 64.0, p->oy[j] / 64.0,
                       p->oox[j], p->ooy[j]));
        }
        j = p->sp[i];
        indent_fn();
        FS_PRINTF(("%3d %02x  %3d %7.2f %7.2f %7.2f %7.2f %7d %7d\n", j,
                   p->f[j], p->onCurve[j], p->x[j] / 64.0, p->y[j] / 64.0,
                   p->ox[j] / 64.0, p->oy[j] / 64.0,
                   p->oox[j], p->ooy[j]));
    }

    indent_fn();
    FS_PRINTF(("PHANTOM's\n"));
    for (j = np; j < np + PHANTOMCOUNT; j++)
    {
        indent_fn();
        FS_PRINTF(("%3d %02x  %3d %7.2f %7.2f %7.2f %7.2f %7d %7d\n", j,
                   p->f[j], p->onCurve[j], p->x[j] / 64.0, p->y[j] / 64.0,
                   p->ox[j] / 64.0, p->oy[j] / 64.0,
                   p->oox[j], p->ooy[j]));
    }
    FS_PRINTF(("\n"));
}
#endif /* NC_DEBUG */
/****************************************************************/
FS_VOID get_glyph_parms(_DS_ TTF *ttf, FS_USHORT gIndex, FS_ULONG *offset, FS_ULONG *length)
{
#ifdef FS_CCC
    int cccFont;

    cccFont = (STIK_FORMAT_CCC(ttf->head->glyphDataFormat)) ||
              (TTF_FORMAT_CCC (ttf->head->glyphDataFormat)) ||
              (STIK_FORMAT_DDD(ttf->head->glyphDataFormat)) ||
              (TTF_FORMAT_DDD (ttf->head->glyphDataFormat)  );
    if ( cccFont )
    {
        BITIO bio;
        FS_LONG numBits = ttf->ccc_info.numBits_LOCA;

        if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
        {
            bio.array = (FS_BYTE *)ttf->loca;
        }
        else
        {
            bio.array = (FS_BYTE *)ttf->loca;
        }
        bio.index = gIndex * numBits;
        bio.max = ( 1 + ttf->maxp->numGlyphs ) * numBits;

        *offset = MTX_BITIO_ReadValue(&bio, numBits);
        *length = MTX_BITIO_ReadValue(&bio, numBits) - *offset;
    }
    else
    {
#endif
        if ( ttf->head->indexToLocFormat == 0  ) /* short format */
        {
            FS_USHORT *p;
            FS_ULONG temp;
            if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
            {
                p = (FS_USHORT *)ttf->loca;
            }
            else
            {
                p = (FS_USHORT *)ttf->loca;
            }

            temp = SWAPW(p[gIndex]);
            *offset = temp << 1;
            temp = SWAPW(p[gIndex + 1]);
            *length = (temp << 1) - *offset;
        }
        else
        {
            if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
            {
                *offset = SWAPL(ttf->loca[gIndex]);
                *length = SWAPL(ttf->loca[gIndex + 1]) - *offset;
            }
            else
            {
                *offset = SWAPL(ttf->loca[gIndex]);
                *length = SWAPL(ttf->loca[gIndex + 1]) - *offset;
            }
        }
#ifdef FS_CCC
    }
#endif

    (void)FS_state_ptr; /*unused*/
}

#define SWAPWINC(a)     SWAPW(*(a)); a++

#define ALMOSTZERO 33

#define FRACT2FIX(n)    (((n) + (1L << 13)) >> 14)

#define NUMBEROFPOINTS(elementPtr)    (elementPtr->nc ? (elementPtr->ep[elementPtr->nc - 1] + 1 + PHANTOMCOUNT) : PHANTOMCOUNT )

#define ISNOTPOWEROF2(n) ((n) & ((n)-1))
#define FITSINAWORD(n)   ((n) < 32768L)

#define STRETCH            2
#define CANTAKESHIFT    0x02000000L

#define ANY_SCALE (WE_HAVE_A_SCALE | WE_HAVE_AN_X_AND_Y_SCALE | WE_HAVE_A_TWO_BY_TWO)


/* #define fsg_MxCopy(a, b)    (*(b) = *(a)) */

/* PRIVATE PROTOTYPES <4> */
FS_VOID fsg_CopyElementBackwards(fnt_ElementType *elementPtr);
static FS_VOID fsg_GetMatrixStretch(FS_FIXED* xStretch,  FS_FIXED* yStretch, transMatrix* trans); /*<8>*/
static FS_BYTE fsg_Identity(transMatrix *matrix);
static int fsg_GetShift(FS_ULONG n);
static FS_VOID fsg_ScaleCVT(fsg_SplineKey *key, FS_LONG numCVT, F26DOT6 *cvt, FS_SHORT *srcCVT);
static FS_VOID fsg_ZeroOutTwilightZone(fsg_SplineKey *key);
static FS_VOID fsg_SetUpProgramPtrs(fsg_SplineKey* key, fnt_GlobalGraphicStateType* globalGS);
static FS_VOID fsg_SetUpTablePtrs(_DS_ fsg_SplineKey* key);
static FS_BYTE fsg_Max45Trick(FS_FIXED x, FS_FIXED y, FS_FIXED* stretch);
static int fsg_CountLowZeros(FS_ULONG n);
static FS_FIXED fsg_MaxAbs(FS_FIXED a, FS_FIXED b);
static int sfnt_UnfoldCurve(fsg_SplineKey* key, sfnt_PackedSplineFormat* charData,
                            FS_LONG* sizeOfInstructions, FS_BYTE** instructionPtr, FS_LONG length);
#ifdef FS_HINTS
static FS_VOID fsg_SnapShotOutline(fsg_SplineKey* key, fnt_ElementType* elem, FS_LONG numPts);
#endif /* FS_HINTS */

/*
* the simplest explaination of how to do a nested composite is
* follow the path until you get some actual data - a simple char
* a) scale that data by scaleFunc() -- the nominal scale
* b) do the hints if any
* c) apply the local transformation if any
* d) apply the local offsets - XY_VALUES are scaled by scaleFunc()
*
* as we go back up the path toward the root character, we
* continue to scale the data by the local transformation and
* apply the local offsets.
*
* There are several problems relating to applying the local
* transform to already hinted data:
* 1) stems are probably no longer grid aligned
* 2) stems are probably no longer integral width
* 3) we're re-scaling ALL the coordinates at each level
*
* But we can formulate this process in an equivelent manner which
* mostly avoids these problems.  Suppose we have a composite
* char 1, which applies off_1 and scale_1 to char 2
* char 2, which applies off_2 and scale_2 to char 3
* char 3, which actually has data.
*
* From the bottom up we have constructed the coordinates
* (scaled(data)*scale_2 + scaled(off_2))*scale_1 + scaled(off_1)
*
* where '*' is matrix multiplication
*
* This is equivelent to
* scaled(data)*scale_2*scale_1 + scaled(off_2)*scale_1 + scaled(off_1)
*
* So if we go down the character and accumulate the scale matricies
* we can get hint the data at it's final size; typically the scaled
* offsets (scaled(off_2)*scale_1) will be rounded to integer; so in
* the final character, the bottom most component will still be grid
* fitted.  Also since we are composing the transformations, we only
* multiply the data by 1 transformation, ever.  Cool.
*/


/* prototypes */
FS_VOID compose_trans(
    FS_FIXED *result,
    FS_FIXED *base,
    FS_FIXED *trans);

FS_VOID init_phantom_points(
    fnt_ElementType *elementPtr,
    sfnt_BBox *bbox,
    fsg_Metrics *metrics);

FS_VOID comp_phantom_points(
    fsg_SplineKey *key,
    fnt_ElementType *elementPtr,
    sfnt_BBox *bbox,
    fsg_Metrics *metrics);

FS_INLINE static FS_VOID scale_piece2(
    ScaleFunc scaleFunc,
    scale_rec *rec,
    F26DOT6 *oop,
    F26DOT6 *p,
    FS_LONG np);

static FS_VOID scale_piece(fsg_SplineKey *key,
                           F26DOT6 *x,
                           F26DOT6 *y,
                           F26DOT6 *oox,
                           F26DOT6 *ooy,
                           FS_LONG np);

FS_VOID scale_char(
    fsg_SplineKey *key,
    fnt_ElementType *elementPtr);

static FS_VOID apply_transformation(fsg_SplineKey *key,
                                    fnt_ElementType *elementPtr,
                                    FS_FIXED *trans);
FS_VOID apply_offsets(_DS_
                      fsg_SplineKey *key,
                      int comps,
                      FS_USHORT flags,
                      FS_SHORT arg1,
                      FS_SHORT arg2,
                      FS_FIXED *trans,
                      fnt_ElementType *elementPtr);
static FS_LONG do_hints(_DS_
                        fsg_SplineKey *key,
                        fnt_ElementType *elementPtr,
                        FS_LONG num,
                        FS_BYTE *ptr,
                        int finalCompositePass,
                        int useHints,
                        fsg_Metrics *metrics);
BITIO *read_comp_parms_ttf(
    fsg_SplineKey *key,
    BITIO *p,
    FS_USHORT *index,
    FS_USHORT *flags,
    FS_SHORT *arg1,
    FS_SHORT *arg2,
    FS_FIXED *trans,
    FS_BOOLEAN *more_comps);

BITIO *read_comp_parms_ccc(_DS_
                           fsg_SplineKey *key,
                           BITIO *p,
                           FS_USHORT *index,
                           FS_USHORT *flags,
                           FS_SHORT *arg1,
                           FS_SHORT *arg2,
                           FS_FIXED *trans,
                           FS_BOOLEAN *more_comps);

BITIO *read_comp_parms(_DS_
                       fsg_SplineKey *key,
                       BITIO *p,
                       FS_USHORT *index,
                       FS_USHORT *flags,
                       FS_SHORT *arg1,
                       FS_SHORT *arg2,
                       FS_FIXED *trans,
                       FS_BOOLEAN *more_comps);

FS_VOID update_sp_ep(int comps, fnt_ElementType *elementPtr);

static FS_VOID advance_elementPtr(fsg_SplineKey *key, fnt_ElementType *p);

BITIO *get_charData_ccc(_DS_
                        BITIO *p,
                        sfnt_PackedSplineFormat *charData);

BITIO *get_charData_ttf(
    BITIO *p,
    sfnt_PackedSplineFormat *charData);

BITIO *get_charData(_DS_
                    BITIO *p,
                    sfnt_PackedSplineFormat *charData);

FS_VOID apply_transMatrix(fsg_SplineKey *key, F26DOT6 *x, F26DOT6 *y, int np, transMatrix *trans);

FS_SHORT get_comp(_DS_
                  fsg_SplineKey *key,
                  FS_USHORT index,
                  FS_FIXED *c_trans,
                  FS_USHORT flags,
                  FS_SHORT arg1,
                  FS_SHORT arg2,
                  FS_FIXED *p_trans,
                  int useHints,
                  fsg_Metrics *metrics);

static FS_VOID set_metrics(fsg_Metrics *metrics, fnt_ElementType *elementPtr, FS_ULONG np);

static FS_VOID get_metrics(fnt_ElementType *elementPtr, fsg_Metrics *metrics, FS_ULONG np);

static FS_VOID update_bearings(fnt_ElementType *elementPtr, fsg_SplineKey *key, FS_ULONG np, FS_BOOLEAN component, int useHints);

/**********************************************************************************/
static FS_VOID set_metrics(fsg_Metrics *metrics, fnt_ElementType *elementPtr, FS_ULONG np)
{
    metrics->lsb_x = elementPtr->x[np + LEFTSIDEBEARING];
    metrics->lsb_y = elementPtr->y[np + LEFTSIDEBEARING];
    metrics->rsb_x = elementPtr->x[np + RIGHTSIDEBEARING];
    metrics->rsb_y = elementPtr->y[np + RIGHTSIDEBEARING];
    metrics->tsb_x = elementPtr->x[np + TOPSIDEBEARING];
    metrics->tsb_y = elementPtr->y[np + TOPSIDEBEARING];
    metrics->bsb_x = elementPtr->x[np + BOTTOMSIDEBEARING];
    metrics->bsb_y = elementPtr->y[np + BOTTOMSIDEBEARING];
}


/**********************************************************************************/

static FS_VOID get_metrics(fnt_ElementType *elementPtr, fsg_Metrics *metrics, FS_ULONG np)
{
    elementPtr->x[np + LEFTSIDEBEARING] = metrics->lsb_x;
    elementPtr->y[np + LEFTSIDEBEARING] = metrics->lsb_y;
    elementPtr->x[np + RIGHTSIDEBEARING] = metrics->rsb_x;
    elementPtr->y[np + RIGHTSIDEBEARING] = metrics->rsb_y;
    elementPtr->x[np + TOPSIDEBEARING] = metrics->tsb_x;
    elementPtr->y[np + TOPSIDEBEARING] = metrics->tsb_y;
    elementPtr->x[np + BOTTOMSIDEBEARING] = metrics->bsb_x;
    elementPtr->y[np + BOTTOMSIDEBEARING] = metrics->bsb_y;
}


/**********************************************************************************/

static FS_VOID update_bearings(fnt_ElementType *elementPtr, fsg_SplineKey *key, FS_ULONG np, FS_BOOLEAN component, int useHints)
{
    fnt_GlobalGraphicStateType* globalGS = GLOBALGSTATE(key);
    FS_LONG lsb, old_lsb, width;
    FS_LONG tsb, height;

    tsb = elementPtr->x[np + TOPSIDEBEARING];
    tsb += 32;
    tsb &= ~63;
    elementPtr->x[np + TOPSIDEBEARING] = tsb;
    elementPtr->x[np + BOTTOMSIDEBEARING] = tsb;

    tsb = elementPtr->y[np + TOPSIDEBEARING];
    tsb += 32;
    tsb &= ~63;
    elementPtr->y[np + TOPSIDEBEARING] = tsb;


    height = LongMulDiv(globalGS->interpScalarY,
                        elementPtr->ooy[np + BOTTOMSIDEBEARING] -
                        elementPtr->ooy[np + TOPSIDEBEARING],
                        key->emResolution);        /* as 16.16 */

    height += (1 << 9);
    height >>= 10;    /* now 26.6 */
    if (useHints)
    {
        height += 32;
        height &= ~63;    /* made integral */
    }
    elementPtr->y[np + BOTTOMSIDEBEARING] = tsb + height;


    /* left and right side bearings */
    lsb = old_lsb = elementPtr->x[np + LEFTSIDEBEARING];
    lsb += 32;
    lsb &= ~63;    /* integral as 26.6 */
    elementPtr->x[np + LEFTSIDEBEARING] = lsb;


    /* if a component, shift entire char by this amount */
    if ( (lsb != old_lsb) && component )
    {
        F26DOT6 *x, dx = lsb - old_lsb;
        FS_LONG n = np;

#ifdef NC_DEBUG
        indent_fn();
        FS_PRINTF(("moving x's by dx=%d %f\n", dx, dx / 64.0));
#endif
        x = elementPtr->x;
        while (n--)
            *x++ += dx;

#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "after !finalCompositePass");
#endif
    }

    width = LongMulDiv(globalGS->interpScalarX,
                       elementPtr->oox[np + RIGHTSIDEBEARING] -
                       elementPtr->oox[np + LEFTSIDEBEARING],
                       key->emResolution); /* 16.16 - as is interpScalar */
    width += (1 << 9);
    width >>= 10;    /* now 26.6 */
    if (useHints)
    {
        width += 32;
        width &= ~63;
    }
    elementPtr->x[np + RIGHTSIDEBEARING] = lsb + width;


#ifdef NC_DEBUG
    dump_elementPtr(elementPtr, "after right side bearing");
#endif

}

/**********************************************************************************/
static fsg_SplineKey* fs_SetUpKey(fs_GlyphInputType* inptr,  FS_USHORT stateBits, int* err)
{
    register fsg_SplineKey* key;

    key = (fsg_SplineKey *)(FS_BYTE *)(inptr->memoryBases[KEY_PTR_BASE]);

    if ( (key->state & stateBits) != stateBits )
    {
        *err = OUT_OFF_SEQUENCE_CALL_ERR;
        return 0;
    }
    key->sfnt = inptr->sfnt;
    *err = NO_ERR;

    return key;
}

/**********************************************************************************/
static FS_LONG max_abs(FS_LONG a, FS_LONG b)
{
    if (a < 0) a = -a;
    if (b < 0) b = -b;
    return (a > b) ? a : b;
}

/**********************************************************************************/
static ScaleFunc compute_scaling(scale_rec *rec, FS_FIXED interpScalar, int emResolution)
{
    FS_FIXED N = interpScalar;
    FS_FIXED D = (FS_FIXED)emResolution << 16;
    int shift;

    /* would like D to be dividable by 2 */
    shift = fsg_CountLowZeros( N | D ) - 1;
    if (shift > 0)
    {
        N >>= shift;
        D >>= shift;
    }

    if ( N < CANTAKESHIFT )
        N <<= fnt_pixelShift;
    else
        D >>= fnt_pixelShift;

    rec->N = N;
    rec->D = D;
    rec->scl = 0;
    if ( FITSINAWORD( N ) )
    {
        shift = fsg_GetShift( D );

        if ( shift >= 0 )                       /* FAST SCALE */
        {
            rec->shift = shift;
            return SFI_fnt_FRound;
        }
        else                                    /* MEDIUM SCALE */
        {
            return SFI_fnt_SRound;
        }
    }
    else                                        /* SLOW SCALE */
    {
        rec->scl = FixDiv(N, D);
        return SFI_fnt_FixRound;
    }
}

/**********************************************************************************/
/*
 * fsg_PrivateFontSpaceSize : This data should remain intact for the life of the sfnt
 *        because function and instruction defs may be defined in the font program
 *        and/or the preprogram.
 */
static FS_ULONG fsg_PrivateFontSpaceSize(fsg_SplineKey *key)
{
    FS_ULONG size;

    key->offset_storage = size = 0;

    /* make sure there are at least 4 storage areas for stik fonts */
    if (key->maxp->maxStorage < 4)
        key->maxp->maxStorage = 4;

    size += sizeof(F26DOT6) * key->maxp->maxStorage;
    FS_ALIGN( size );

    key->offset_controlValues = size;
    key->cvtCount = ((TTF *)(key->sfnt->lfnt->fnt))->num_cvt;
    size += sizeof(F26DOT6) * key->cvtCount;
    FS_ALIGN( size );

    return( size );
}

/*lint -e578  Warning 578: Declaration of symbol '' hides symbol */

/**********************************************************************************/
/*
 * This guy asks for memory for points, instructions, fdefs and idefs.
 */
FS_LONG fs_NewSfnt(fs_GlyphInputType *inputPtr, fs_GlyphInfoType *outputPtr)
{
    int error;
    TTF *ttf = (TTF *)(inputPtr->sfnt->lfnt->fnt);
    TTF_HEAD *fontHead = ttf->head;
    TTF_HHEA *horiHead = ttf->hhea;

    fsg_SplineKey* key = fs_SetUpKey(inputPtr, FS_INITIALIZED, &error);
    if (!key)
        return error;

    key->emResolution               = fontHead->unitsPerEm;
    outputPtr->emResolution         = key->emResolution;
    key->fontFlags                  = fontHead->flags;

    key->numberOf_LongHorMetrics = horiHead->numberOf_LongHorMetrics;
    key->maxp = ttf->maxp;

    outputPtr->memorySizes[PRIVATE_FONT_SPACE_BASE] = fsg_PrivateFontSpaceSize(key);
    outputPtr->memorySizes[WORK_SPACE_BASE]         = fsg_WorkSpaceSetOffsets(key);

    key->state = (FS_USHORT)(FS_INITIALIZED | FS_NEWSFNT);
    key->scanControl = 0;

    /*
     *    Can't run font program yet, we don't have any memory for the graphic state
     *    Mark it to be run in NewTransformation.
     */
    key->executeFontPgm = 1;

    return NO_ERR;
}

/**********************************************************************************/
/*
 *    All this guy does is record FDEFs and IDEFs, anything else is ILLEGAL
 */
static FS_LONG fsg_RunFontProgram(_DS_ fsg_SplineKey* key )
{ 
#define ITYPE_UD_FUNCTIONS    1
#define GENERIC_UD_FUNCTIONS  2
    
    FS_LONG result = NO_ERR;
    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);
    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);

    globalGS->instrDefCount = 0;        /* none allocated yet, always do this, even if there's no fontProgram */
       
#ifdef FS_HINTS
#ifndef GLOBAL_STATIC_FUNCTION_PTRS_OK
    /* initialize hinting function pointers for systems that do not */
    /* allow global static function pointers                        */
    fnt_initInterpFuncs();
#endif
    /* Set user-defined opcodes (0xa2-0xaf) to ITYPE's hinting functions - functions as default */
    globalGS->udFuncType = ITYPE_UD_FUNCTIONS;
    set_UD_functions(ITYPE_UD_FUNCTIONS);
#endif

    if (ttf->functionDefs && ttf->num_fpgm)
    {
        globalGS->pgmIndex = FONTPROGRAM;
        fsg_SetUpProgramPtrs(key, globalGS);
        fsg_SetUpTablePtrs(_PS_ key);
        globalGS->glyphProgram = 0;

#ifdef FS_HINTS
        result = fnt_Execute(key->elementInfoRec.interpreterElements,
                             (FS_BYTE *)(globalGS->pgmList[FONTPROGRAM]),
                             (FS_BYTE *)(globalGS->pgmList[FONTPROGRAM]) + ttf->num_fpgm,
                             globalGS);
        
        /* If fpgm table references a user-defined opcode (0xa2-0xaf) by IDEFs, set user-defined */
        /* opcodes func. to generic fnt_IdefPatch() functions */
        if (globalGS->udFuncType == GENERIC_UD_FUNCTIONS)
            set_UD_functions(GENERIC_UD_FUNCTIONS);
#endif
        key->executeFontPgm = 0;
        return result;
    }
    return NO_ERR;
}

/**********************************************************************************/
/* set the scale goop for a new transformation */
FS_LONG fs_NewTransformation(_DS_ fs_GlyphInputType *inputPtr, fs_GlyphInfoType *outputPtr )
{
    register fsg_SplineKey         *key;
    int                            error;

    key = fs_SetUpKey(inputPtr, FS_INITIALIZED | FS_NEWSFNT, &error);
    if (!key)
        return error;

    /* we're about to delete the <inputPtr> in caller */
    key->memoryBases[0] = inputPtr->memoryBases[0];
    key->memoryBases[1] = inputPtr->memoryBases[1];
    key->memoryBases[2] = inputPtr->memoryBases[2];
    key->memoryBases[3] = inputPtr->memoryBases[3];

    key->currentTMatrix = inputPtr->param.newtrans.matrix;
    key->fixedPointSize = inputPtr->param.newtrans.pointSize;
    key->pixelDiameter    = inputPtr->param.newtrans.pixelDiameter;

    /* fold resolutions and pointSize into matrix */
    {
        FS_SHORT xres = inputPtr->param.newtrans.xResolution;
        FS_SHORT yres = inputPtr->param.newtrans.yResolution;
        FS_FIXED scl;
        transMatrix *trans = &key->currentTMatrix;

        scl = ShortMulDiv(key->fixedPointSize, xres, POINTSPERINCH);
        trans->transform[0][0] = FixMul(trans->transform[0][0], scl);
        trans->transform[1][0] = FixMul(trans->transform[1][0], scl);
        trans->transform[2][0] = FixMul(trans->transform[2][0], scl);

        scl = ShortMulDiv(key->fixedPointSize, yres, POINTSPERINCH);
        trans->transform[0][1] = FixMul(trans->transform[0][1], scl);
        trans->transform[1][1] = FixMul(trans->transform[1][1], scl);
        trans->transform[2][1] = FixMul(trans->transform[2][1], scl);

#ifdef NC_DEBUG
        dump_matrix("", *trans);
#endif
    }

    /* set up different x/y scaling records */
    {
        transMatrix *trans = &key->currentTMatrix;
        fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);

        globalGS->metricScalarX = max_abs(trans->transform[0][0], trans->transform[0][1]);
        globalGS->metricScalarY = max_abs(trans->transform[1][0], trans->transform[1][1]);

        if (key->fontFlags & USE_INTEGER_SCALING)
        {
            globalGS->interpScalarX = (globalGS->metricScalarX + 0x8000) & 0xFFFF0000;
            globalGS->interpScalarY = (globalGS->metricScalarY + 0x8000) & 0xFFFF0000;
        }
        else
        {
            globalGS->interpScalarX = globalGS->metricScalarX;
            globalGS->interpScalarY = globalGS->metricScalarY;
        }
        globalGS->scaleFuncX = compute_scaling(&globalGS->scaleX, globalGS->interpScalarX, key->emResolution);
        globalGS->scaleFuncY = compute_scaling(&globalGS->scaleY, globalGS->interpScalarY, key->emResolution);

        /*  nb: this makes cvtStretchX and cvtStretchY <= 1 */
        if (globalGS->interpScalarX >= globalGS->interpScalarY)
        {
            globalGS->scaleFuncCVT = globalGS->scaleFuncX;
            globalGS->scaleCVT = globalGS->scaleX;
            globalGS->cvtStretchX = FIXED_ONE;
            globalGS->cvtStretchY = FixDiv(globalGS->interpScalarY, globalGS->interpScalarX);
            key->metricScalar = globalGS->metricScalarX;
            key->interpScalar = globalGS->interpScalarX;
        }
        else
        {
            globalGS->scaleFuncCVT = globalGS->scaleFuncY;
            globalGS->scaleCVT = globalGS->scaleY;
            globalGS->cvtStretchX = FixDiv(globalGS->interpScalarX, globalGS->interpScalarY);
            globalGS->cvtStretchY = FIXED_ONE;
            key->metricScalar = globalGS->metricScalarY;
            key->interpScalar = globalGS->interpScalarY;
        }

#ifdef NC_DEBUG
        FS_PRINTF(("globalGS->metricScalarX = %.5f\n", globalGS->metricScalarX / 65536.0));
        FS_PRINTF(("globalGS->metricScalarY = %.5f\n", globalGS->metricScalarY / 65536.0));
#endif
    }

    fsg_InitInterpreterTrans(key);        /* get premultipliers if any, also called in sfnt_ReadSFNT */

    /*
     *    This guy defines FDEFs and IDEFs.  The table is optional
     */
    if (key->executeFontPgm)
    {
        if ((error = fsg_SetDefaults(key)) != 0)
            return error;
        if ((error = fsg_RunFontProgram(_PS_ key)) != 0)
            return error;
    }

    /* postpone as long as possible */
    /* Run the pre program and scale the control value table MUCH LATER */
    key->executePrePgm = 1;

    key->state = (FS_USHORT)(FS_INITIALIZED | FS_NEWSFNT | FS_NEWTRANS);

    outputPtr->scaledCVT = (F26DOT6 *)((FS_BYTE *)key->memoryBases[PRIVATE_FONT_SPACE_BASE] +
                                                    key->offset_controlValues);

    return NO_ERR;
}
/*lint +e578  Warning 578: Declaration of symbol '' hides symbol */

/****************************************************************/
FS_VOID compose_trans(FS_FIXED *result, FS_FIXED *base, FS_FIXED *trans)
{
    FS_FIXED temp[4]; /* in case result == base */

    /* if  x' = base[0]*x + base[1]*y
    * and  y' = base[2]*x + base[3]*y
    *
    * we want
    * x" = trans[0]*x' + trans[1]*y'
    * y" = trans[2]*x' + trans[3]*y'
    *
    * so
    * x" = trans[0]*(base[0]*x + base[1]*y) + trans[1]*(base[2]*x + base[3]*y)
    * y" = trans[2]*(base[0]*x + base[1]*y) + trans[3]*(base[2]*x + base[3]*y)
    *
    * rearrange
    * x" = (trans[0]*base[0]+trans[1]*base[2])*x + (trans[0]*base[1]+trans[1]*base[3])*y
    * y" = (trans[2]*base[0]+trans[3]*base[2])*x + (trans[2]*base[1]+trans[3]*base[3])*y
    *
    * result[0] = (trans[0]*base[0]+trans[1]*base[2])
    * result[1] = (trans[0]*base[1]+trans[1]*base[3])
    * result[2] = (trans[2]*base[0]+trans[3]*base[2])
    * result[3] = (trans[2]*base[1]+trans[3]*base[3])
    *
    * These are all really FixMul's ... but special cases are useful to note
    */

    if (base[1] == 0 && base[2] == 0)
    {
        /* diagonal base */
        if (trans[1] == 0 && trans[2] == 0)
        {
            /* diagonal base and diagonal trans */
            temp[0] = FixMul(trans[0], base[0]);
            temp[1] = 0;
            temp[2] = 0;
            temp[3] = FixMul(trans[3], base[3]);
        }
        else
        {
            /* diagonal base and general trans */
            temp[0] = FixMul(trans[0], base[0]);
            temp[1] = FixMul(trans[1], base[3]);
            temp[2] = FixMul(trans[2], base[0]);
            temp[3] = FixMul(trans[3], base[3]);
        }
    }
    else    /* general base */
    {
        if (trans[1] == 0 && trans[2] == 0)
        {
            /* general base and diagonal trans */
            temp[0] = FixMul(trans[0], base[0]);
            temp[1] = FixMul(trans[0], base[1]);
            temp[2] = FixMul(trans[3], base[2]);
            temp[3] = FixMul(trans[3], base[3]);
        }
        else
        {
            /* general base and general trans */
            temp[0] = FixMul(trans[0], base[0]) + FixMul(trans[1], base[2]);
            temp[1] = FixMul(trans[0], base[1]) + FixMul(trans[1], base[3]);
            temp[2] = FixMul(trans[2], base[0]) + FixMul(trans[3], base[2]);
            temp[3] = FixMul(trans[2], base[1]) + FixMul(trans[3], base[3]);
        }
    }
    SYS_MEMCPY(result, temp, 16);
}


/****************************************************************/
/* initialize phantom points (in FRU's) for scaler */
FS_VOID init_phantom_points(fnt_ElementType *elementPtr, sfnt_BBox *bbox, fsg_Metrics *metrics)
{
    FS_SHORT nc = elementPtr->nc;
    FS_LONG np = nc ? (1 + elementPtr->ep[nc - 1] - elementPtr->sp[0]) : 0;
    F26DOT6 xMinMinusLSB = bbox->xMin - metrics->lsb;
    F26DOT6 yMaxPlusTSB = metrics->ah;
    F26DOT6 mid;

    /* horizontal phantoms -- on the baseline */
    elementPtr->ooy[np + LEFTSIDEBEARING] = 0;
    elementPtr->ooy[np + ORIGINPOINT] = 0;
    elementPtr->ooy[np + LEFTEDGEPOINT] = 0;
    elementPtr->ooy[np + RIGHTSIDEBEARING] = 0;

    elementPtr->oox[np + LEFTSIDEBEARING] = xMinMinusLSB;
    elementPtr->oox[np + ORIGINPOINT] = xMinMinusLSB;
    elementPtr->oox[np + LEFTEDGEPOINT] = bbox->xMin;
    elementPtr->oox[np + RIGHTSIDEBEARING] = xMinMinusLSB + metrics->aw;

    /* vertical phantoms -- on the character midline ??? */
    mid = (elementPtr->oox[np + LEFTSIDEBEARING] + elementPtr->oox[np + RIGHTSIDEBEARING]) / 2;
    elementPtr->oox[np + TOPSIDEBEARING] = mid;
    elementPtr->oox[np + BOTTOMSIDEBEARING] = mid;
    elementPtr->oox[np + TOPORIGINPOINT] = mid;
    elementPtr->oox[np + TOPEDGEPOINT] = mid;

    elementPtr->ooy[np + TOPSIDEBEARING] = yMaxPlusTSB;
    elementPtr->ooy[np + BOTTOMSIDEBEARING] = yMaxPlusTSB + metrics->ah;
    elementPtr->ooy[np + TOPORIGINPOINT] = yMaxPlusTSB;
    elementPtr->ooy[np + TOPEDGEPOINT] = bbox->yMax;


#ifdef NC_DEBUG
    dump_elementPtr(elementPtr, "after init_phantom_points");
#endif

}
/****************************************************************/
/* reset composites phantom points from initial parameters, and scale them */
FS_VOID comp_phantom_points(
    fsg_SplineKey *key,
    fnt_ElementType *elementPtr,
    /* FS_FIXED *trans, */ /* qwu, 07-03-06 */
    sfnt_BBox *bbox,
    fsg_Metrics *metrics)
{
    FS_SHORT nc = elementPtr->nc;
    FS_LONG np = nc ? (1 + elementPtr->ep[nc - 1] - elementPtr->sp[0]) : 0;
    F26DOT6 xMinMinusLSB = bbox->xMin - metrics->lsb;
    F26DOT6 yMaxPlusTSB = metrics->ah;
    F26DOT6 mid;

    /* horizontal phantoms */
    elementPtr->ooy[np + LEFTSIDEBEARING] = 0;
    elementPtr->ooy[np + ORIGINPOINT] = 0;
    elementPtr->ooy[np + LEFTEDGEPOINT] = 0;
    elementPtr->ooy[np + RIGHTSIDEBEARING] = 0;

    elementPtr->oox[np + LEFTSIDEBEARING] = xMinMinusLSB;
    elementPtr->oox[np + ORIGINPOINT] = xMinMinusLSB;
    elementPtr->oox[np + LEFTEDGEPOINT] = bbox->xMin;
    elementPtr->oox[np + RIGHTSIDEBEARING] = xMinMinusLSB + metrics->aw;

    /* vertical phantoms -- on the character midline ??? */
    mid = (elementPtr->oox[np + LEFTSIDEBEARING] + elementPtr->oox[np + RIGHTSIDEBEARING]) / 2;
    elementPtr->oox[np + TOPSIDEBEARING] = mid;
    elementPtr->oox[np + BOTTOMSIDEBEARING] = mid;
    elementPtr->oox[np + TOPORIGINPOINT] = mid;
    elementPtr->oox[np + TOPEDGEPOINT] = mid;

    elementPtr->ooy[np + TOPSIDEBEARING] = yMaxPlusTSB;
    elementPtr->ooy[np + BOTTOMSIDEBEARING] = yMaxPlusTSB + metrics->ah;
    elementPtr->ooy[np + TOPORIGINPOINT] = yMaxPlusTSB;
    elementPtr->ooy[np + TOPEDGEPOINT] = bbox->yMax;

    /* now need to scale them */
    scale_piece(key,
                elementPtr->x + np,
                elementPtr->y + np,
                elementPtr->oox + np,
                elementPtr->ooy + np,
                PHANTOMCOUNT);

#ifdef NC_DEBUG
    dump_elementPtr(elementPtr, "after comp_phantom_points");
#endif

}
/****************************************************************/
FS_INLINE static FS_VOID scale_piece2(ScaleFunc scaleFunc, scale_rec *rec, F26DOT6 *oop, FS_LONG *p, FS_LONG np)
{
    FS_LONG N = rec->N;
    FS_LONG D = rec->D;
    FS_LONG Dover2 = D >> 1;
    FS_FIXED scl = rec->scl;
    int shift = rec->shift;
    FS_LONG V;

    if (scaleFunc == SFI_fnt_FRound)
    {
        while (np--)
        {
            V = *oop++;
            FROUND(V, N, Dover2, shift);
            *p++ = V;
        }
    }
    else if (scaleFunc == SFI_fnt_SRound)
    {
        while (np--)
        {
            V = *oop++;
            SROUND( V, N, D, Dover2 );
            *p++ = V;
        }
    }
    else
    {
        while (np--)
        {
            *p++ = FixMul(scl, *oop++);
        }
    }
}
/****************************************************************/
/* apply global scale function and accumulated local transformations */
static FS_VOID scale_piece(fsg_SplineKey *key,
                           F26DOT6 *x,
                           F26DOT6 *y,
                           F26DOT6 *oox,
                           F26DOT6 *ooy,
                           FS_LONG np)
{
    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);

    scale_piece2(globalGS->scaleFuncX, &globalGS->scaleX, oox, x, np);
    scale_piece2(globalGS->scaleFuncY, &globalGS->scaleY, ooy, y, np);

}
/****************************************************************/
FS_VOID scale_char(fsg_SplineKey *key, fnt_ElementType *elementPtr)
{
    FS_LONG nc, np;
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    F26DOT6 *oox = elementPtr->oox;
    F26DOT6 *ooy = elementPtr->ooy;

    /* point count */
    nc = elementPtr->nc;
    if (nc)
        np = 1 + PHANTOMCOUNT + elementPtr->ep[nc - 1] - elementPtr->sp[0];
    else
        np = PHANTOMCOUNT;

    /* scale them */
    scale_piece(key, x, y, oox, ooy, np);

#ifdef NC_DEBUG
    dump_elementPtr(elementPtr, "after 'scale_char'");
#endif
}

/****************************************************************/
FS_INLINE static F26DOT6 scale_26_6_by_fix(F26DOT6 z, FS_FIXED s)
{
    /* 1f 2.16 scale factors are in the range (-2.0,+2.0) == (-131072L,+131072L)  */
    /* if the coordinate is in the interval (-16384,+16384) == (-256,+256) pixels */
    /* then we can do the multiply in +/- 31 bits -- we don't need to call FixMul */
    if (-16384L <= z && z <= 16384L && -131072L <= s && s <= 131072L)
    {
        z *= s;
        if (z > 0)
            z += 32768;
        else
            z += 32767;
        z >>= 16;
    }
    else
        z = FixMul(z, s);

    return z;
}

/****************************************************************/
static FS_VOID apply_transformation(fsg_SplineKey *key, fnt_ElementType *elementPtr, FS_FIXED *trans)
{
    FS_FIXED xScale, yScale;
    F26DOT6* x, *ox;
    F26DOT6* y, *oy;
    FS_SHORT numPts;
    F26DOT6 xx;
    F26DOT6 yy;
    SENV *senv = (SENV *)((SFNT *)(key->sfnt)->senv);
    FS_FIXED t0, t1, t2, t3;

    if (elementPtr->nc == 0)
        return;
    if (key->localTIsIdentity)
        return;

    t0 = trans[0];
    t1 = trans[1];
    t2 = trans[2];
    t3 = trans[3];
    x = elementPtr->x;
    y = elementPtr->y;
    ox = elementPtr->ox;
    oy = elementPtr->oy;

    xScale = key->tInfo.x;
    yScale = key->tInfo.y;

    if ( senv->vanilla )
    {
        if ( (xScale >> 16) == senv->xppm )
            xScale = FIXED_ONE;
        else
            xScale = FixDiv(xScale, senv->xppm << 16);
        if ( (yScale >> 16) == senv->yppm )
            yScale = FIXED_ONE;
        else
            yScale = FixDiv(yScale, senv->yppm << 16);
    }
    else
    {
        transMatrix *transf = &key->currentTMatrix;
        FS_FIXED* matrix = &transf->transform[0][0];
        FS_FIXED xf, yf;

        xf = *matrix++;
        yf = *matrix++;
        if (xf < 0) xf = -xf;
        if (yf < 0) yf = -yf;
        if (xf < yf)
        {
            if (yf >= FIXED_ONE)
                xScale = xScale / (yf >> 16 );
        }
        else
        {
            if (xf >= FIXED_ONE)
                xScale = xScale / (xf >> 16 );
        }

        matrix++;
        xf = *matrix++;
        yf = *matrix;
        if (xf < 0) xf = -xf;
        if (yf < 0) yf = -yf;
        if (xf < yf)
        {
            if (yf >= FIXED_ONE)
                yScale = yScale / (yf >> 16 );
        }
        else
        {
            if (xf >= FIXED_ONE)
                yScale = yScale / (xf >> 16 );
        }
    }

    numPts = NUMBEROFPOINTS(elementPtr);

    if ( xScale == 0L || yScale == 0L )
    {
        while (numPts--)
            *x++ = *y++ = 0;
    }
    else
    {
        if (xScale == FIXED_ONE && yScale == FIXED_ONE && t1 == 0 && t2 == 0)
        {
            int np = numPts;
            /* usual vanilla case --  diagonal matrix is easy */
            while (np--)
            {
                xx = *x;
                yy = *y;
                *x++ = scale_26_6_by_fix(xx, t0);
                *y++ = scale_26_6_by_fix(yy, t3);
            }
            SYS_MEMCPY(ox, elementPtr->x, numPts * 4);
            SYS_MEMCPY(oy, elementPtr->y, numPts * 4);
        }
        else if (xScale == FIXED_ONE && yScale == FIXED_ONE)
        {
            int np = numPts;
            /* generic 2x2 -- almost never happens */
            while (np--)
            {
                xx = *x;
                yy = *y;
                *x++ = scale_26_6_by_fix(xx, t0) + scale_26_6_by_fix(yy, t2);
                *y++ = scale_26_6_by_fix(xx, t1) + scale_26_6_by_fix(yy, t3);
            }
            SYS_MEMCPY(ox, elementPtr->x, 4 * numPts);
            SYS_MEMCPY(oy, elementPtr->y, 4 * numPts);
        }
        else
        {
            int np = numPts;
            while (np--)
            {
                xx = FixDiv( *x, xScale );
                yy = FixDiv( *y, yScale );
                *x++ = scale_26_6_by_fix(xx, t0) + scale_26_6_by_fix(yy, t2);
                *y++ = scale_26_6_by_fix(xx, t1) + scale_26_6_by_fix(yy, t3);
            }
            SYS_MEMCPY(ox, elementPtr->x, 4 * numPts);
            SYS_MEMCPY(oy, elementPtr->y, 4 * numPts);
        }
    }

    {
        /* if the transformation is a x-mirror, swap the LEFT and RIGHT side bearings */
        F26DOT6 t;
        unsigned s1, s2;
        int lsb = NUMBEROFPOINTS(elementPtr) - PHANTOMCOUNT + LEFTSIDEBEARING;
        int rsb = NUMBEROFPOINTS(elementPtr) - PHANTOMCOUNT + RIGHTSIDEBEARING;

        s1 = elementPtr->x[rsb] >= 0;
        s2 = elementPtr->oox[rsb] >= 0;
        if (s1 ^ s2)
        {
            x = elementPtr->x;
            y = elementPtr->y;
            t = x[rsb];
            x[rsb] = x[lsb];
            x[lsb] = t;
            t = y[rsb];
            y[rsb] = y[lsb];
            y[lsb] = t;
        }
    }
}

/****************************************************************/
/* calculate and apply offsets to a component/composite/character */
FS_VOID apply_offsets(_DS_
                      fsg_SplineKey *key,
                      int comps,
                      FS_USHORT flags,
                      FS_SHORT arg1,
                      FS_SHORT arg2,
                      FS_FIXED *trans,
                      fnt_ElementType *elementPtr)
{
    F26DOT6 *x = elementPtr->x;
    F26DOT6 *y = elementPtr->y;
    F26DOT6 *ox = elementPtr->ox;
    F26DOT6 *oy = elementPtr->oy;
    FS_LONG nc, np;
    FS_FIXED dx, dy;

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("apply_offsets(0x%04x, %d, %d, [%.6f %.6f %.6f %.6f]\n", flags, arg1, arg2,
               trans[0] / 65536.0, trans[1] / 65536.0, trans[2] / 65536.0, trans[3] / 65536.0));
#endif

    if (flags & ARGS_ARE_XY_VALUES)
    {
        fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);

        if ( (arg1 == 0) && (arg2 == 0) )
        {
            /* common case */
            nc = elementPtr->nc;
            np = nc ? (1 + PHANTOMCOUNT + elementPtr->ep[elementPtr->nc - 1] + elementPtr->sp[0]) : 0;
            SYS_MEMCPY(ox, elementPtr->x, 4 * np);
            SYS_MEMCPY(oy, elementPtr->y, 4 * np);
            return;
        }
        else
        {

            dx = ScaleFuncCall(globalGS->scaleFuncX, &globalGS->scaleX, (FS_LONG)arg1);
            dy = ScaleFuncCall(globalGS->scaleFuncY, &globalGS->scaleY, (FS_LONG)arg2);

            /* apply (accumulated) trans */
            if (trans[1] == 0 && trans[2] == 0)
            {
                /* diagonal trans */
                dx = FixMul(dx, trans[0]);
                dy = FixMul(dy, trans[3]);
            }

            if (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON)
            {
                dx = (dx + 32) & ~63;
                dy = (dy + 32) & ~63;
            }
            else
            {
                if (flags & ROUND_XY_TO_GRID)    /* this is the fix of ITP-325 */
                {
                    dx = (dx + 32) & ~63;
                    dy = (dy + 32) & ~63;
                }
            }
        }

    }
    else /* anchor points */
    {
        if (comps == 0)
            /* can't use anchor points from the previous
            * component if there are no previous components */
            return;    /* ? should this be an error ? */

        /* number of points in previous component */
        np = 1 + elementPtr->ep[-1];

        /* offsets between corresponding points */
        dx = elementPtr->x[arg1 - np] - elementPtr->x[arg2];
        dy = elementPtr->y[arg1 - np] - elementPtr->y[arg2];
    }

    /* either way ... add the offsets */
    nc = elementPtr->nc;
    np = nc ? (1 + PHANTOMCOUNT + elementPtr->ep[elementPtr->nc - 1] + elementPtr->sp[0]) : 0;

    nc = np;
    while (nc--)
    {
        *x++ += dx;
        *y++ += dy;
    }
    SYS_MEMCPY(ox, elementPtr->x, 4 * np);
    SYS_MEMCPY(oy, elementPtr->y, 4 * np);

#ifdef NC_DEBUG
    dump_elementPtr(elementPtr, "after my 'apply_offsets'");
#endif
}

/****************************************************************/
/* handle USE_MY_METRICS and instructions and such */
static FS_LONG do_hints(_DS_
                 fsg_SplineKey *key,
                 fnt_ElementType *elementPtr,
                 FS_LONG num,
                 FS_BYTE *ptr,
                 int finalCompositePass,
                 int useHints,
                 fsg_Metrics *metrics)
{
    fnt_GlobalGraphicStateType* globalGS = GLOBALGSTATE(key);
    FS_SHORT nc = elementPtr->nc;
    FS_ULONG result = 0, np;
#ifdef FS_HINTS
    TTF *ttf = (TTF *)(FS_VOID *)(STATE.cur_lfnt->fnt);
    FS_BYTE AA[3] = {0xb0, 0, 0x7f}; /* pushb[000],0,0x7F */
#ifdef FS_EDGE_HINTS
    FS_BYTE MAZAA[3] = {0xb0, 5, 0x7f}; /* pushb[000],5,0x7F */
    FS_BYTE BAZAA[3] = {0xb0, 8, 0x7f}; /* pushb[000],5,0x7F */
#endif /* FS_EDGE_HINTS */
#endif /* FS_HINTS */

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("do_hints(num=%d useHints=%d,finalCompositePass=%d)\n", num, useHints, finalCompositePass));
    dump_elementPtr(elementPtr, "on entry to 'do_hints'");
#endif

    /* reset default parameters for glyphs */
    globalGS->localParBlock = globalGS->defaultParBlock;

    /* number of points w/o the phantoms */
    np = nc ? (1 + elementPtr->ep[nc - 1] - elementPtr->sp[0]) : 0;

    /* use a previous component's metrics */
    if (metrics->useMyMetrics && finalCompositePass)
    {
        get_metrics(elementPtr, metrics, np);
        metrics->useMyMetrics = 0;
    }

    if (/*useHints && */!finalCompositePass)
        update_bearings(elementPtr, key, np, 1, useHints);

    /* save my metrics for later */
    if (metrics->useMyMetrics && !finalCompositePass)
    {
        set_metrics(metrics, elementPtr, np);


#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "after useMyMetrics");
#endif
    }

#ifdef FS_HINTS
#ifdef FS_EDGE_HINTS
    if (useHints && (num == 0))
    {
        /* only apply at 90 degree rotations */
        if ( (STATE.cur_sfnt->user_scale[1]==0 && STATE.cur_sfnt->user_scale[2]==0) ||
             (STATE.cur_sfnt->user_scale[0]==0 && STATE.cur_sfnt->user_scale[3]==0) )
        {
            if (globalGS->maz_data.gridFitType == ADF_GRID_FIT_MAZ_PIXEL)
            {
                if (((finalCompositePass == 1) && !STATE.outl_char) ||
                    ((finalCompositePass == 0) && !STATE.outl_char && (STATE.level == 0) ))
                {
                    ptr = MAZAA;
                    num = 3;
                    STATE.any_hints |= OUTL_FLAGS_GRIDFITTYPE_MAZ;
                }
            }
            else if ((globalGS->maz_data.gridFitType == ADF_GRID_FIT_BAZ_PIXEL) &&
                     (STATE.level == 0))
            {
                ptr = BAZAA;
                num = 3;
                STATE.any_hints |= OUTL_FLAGS_GRIDFITTYPE_BAZ;
            }
        }
    }

#endif /* FS_EDGE_HINTS */

    if (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON)
    {
        FS_SHORT format = ttf->head->glyphDataFormat;
        int is_stik = (STATE.cur_lfnt->fontflags & FONTFLAG_STIK);
        int offline_font = ( format == OFFLINE_STIK_FORMAT ) ||
                           ( format == OFFLINE_CCC_FORMAT );

        if (is_stik && useHints && (num == 0) &&
            (finalCompositePass == 0) && !STATE.outl_char)
        {
            if (offline_font)
            {
                ptr = AA;
                num = 3;
            }
        }
    }

    if (useHints && num)
    {
        if ( finalCompositePass )
        {
            fsg_SnapShotOutline( key, elementPtr, np );
        }
        globalGS->pgmIndex = PREPROGRAM;
        fsg_SetUpProgramPtrs(key, globalGS);
        fsg_SetUpTablePtrs(_PS_ key);
        globalGS->glyphProgram = 1;
        globalGS->sfnt = STATE.cur_sfnt;

#ifdef FS_STIK

        /* stuff the storage area with useful stuff */
        if (STATE.cur_lfnt->fontflags & FONTFLAG_STIK)
        {
            if (!STATE.outl_char)
            {
                if (!(STATE.flags & FLAGS_GRAYSCALE) ||
                    (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON))
                {
                    /* use integer stroke width for autohinting */
                    int sw;
                    sw = FS_ROUND(STATE.lpm * FS_NORMAL_STROKE_PCT);
                    sw = MAX(1, sw);
                    /* pseudo-bold does NOT effect the autohinter's x stroke width */
                    globalGS->store[0] = sw << 6;    /* x stroke width */
                    globalGS->store[1] = sw << 6;    /* y stroke width */
                }
                else
                {
                    /* use fractional stroke width for autohinting */
                    FS_FIXED sw = STATE.lpm * STATE.stroke_pct;
                    /* pseudo-bold does NOT effect the autohinter's x stroke width */
                    globalGS->store[0] = FIXEDTODOT6(sw);    /* x stroke width */
                    globalGS->store[1] = FIXEDTODOT6(sw);    /* y stroke width */
                }
            }
        }
#endif /* FS_STIK */

#ifdef FS_HINTS
        globalGS->rtgah_data.fontflags = STATE.cur_lfnt->fontflags;
        globalGS->rtgah_data.stateflags = STATE.flags;
        globalGS->rtgah_data.rtgah_suitable = STATE.cur_sfnt->rtgah_suitable;
        globalGS->rtgah_data.fnt_type = STATE.cur_lfnt->fnt_type;
#endif

#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "before fnt_Execute");
#endif

        STATE.any_hints |= OUTL_FLAGS_ANYHINTS;

#ifdef FS_EDGE_HINTS
        globalGS->maz_data.MAZmethodHorz = MAZTOPDOWN;
        globalGS->maz_data.MAZmethodVert = MAZTOPDOWN;
        globalGS->maz_data.MAZdeltaHorz = APPLYMAZDELTASBEFORE;
        globalGS->maz_data.MAZdeltaVert = APPLYMAZDELTASBEFORE;
        globalGS->maz_data.numHhints = 0;
        globalGS->maz_data.numVhints = 0;
        globalGS->maz_data.numXdeltas = 0;
        globalGS->maz_data.numYdeltas = 0;
        globalGS->maz_data.foundacollisionhorz = 0;
        globalGS->maz_data.foundacollisionvert = 0;
        {
            TTF_OS2  *pos2 = (TTF_OS2 *)ttf->os2;
            if (pos2)
            {
                FS_USHORT usWeightClass = ttf->os2->usWeightClass;
                if (usWeightClass < 500)
                    globalGS->maz_data.maxWidth = 0x1999; /* 0.099990845 */
                else if (usWeightClass < 700)
                    globalGS->maz_data.maxWidth = 0x2320; /* 0.137207031 */
                else if (usWeightClass < 900)
                    globalGS->maz_data.maxWidth = 0x3720; /* 0.215332031 */
                else
                    globalGS->maz_data.maxWidth = 0x4620; /* 0.273925781 */
            }
            else /* if no OS2 table */
                globalGS->maz_data.maxWidth = 0x4620; /* 0.273925781 */
        }
#endif


        if (useHints && ptr != 0)
        {
            result = fnt_Execute(key->elementInfoRec.interpreterElements,
                                 ptr, ptr + num, globalGS);
        }

#if defined(FS_EDGE_HINTS) && defined(FS_GRIDVIEWER)
        STATE.gridViewerFlag = 0;
        if (globalGS->maz_data.foundacollisionhorz)
        {
            STATE.gridViewerFlag = 1;
        }
        if (globalGS->maz_data.foundacollisionvert)
        {
            STATE.gridViewerFlag |= 2;
        }
#endif

#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "after fnt_Execute");
#endif
    }
#else
    ptr = ptr; /* avoid compiler warnings */
    num = num;
    FS_state_ptr = FS_state_ptr;
#endif    /* FS_HINTS */

    return result;
}
/****************************************************************/
/* this is the TTF version */
BITIO *read_comp_parms_ttf(
    fsg_SplineKey *key,
    BITIO *bio,
    FS_USHORT *index,
    FS_USHORT *flags,
    FS_SHORT *arg1,
    FS_SHORT *arg2,
    FS_FIXED *trans,
    FS_BOOLEAN *more_comps)
{
    FS_USHORT f;
    FS_USHORT *p = (FS_USHORT *)bio->array;

    *flags = f = SWAPWINC(p);
    *index = SWAPWINC(p);

    /* args */
    if (f & ARG_1_AND_2_ARE_WORDS)
    {
        *arg1 = (FS_SHORT) SWAPWINC(p);
        *arg2 = (FS_SHORT) SWAPWINC(p);
    }
    else
    {
        FS_TINY *cp = (FS_TINY *)p;
        *arg1 = (FS_SHORT) * cp++;
        *arg2 = (FS_SHORT) * cp;
        p++;
    }

    /* scale */
    if (f & WE_HAVE_A_SCALE)
    {
        FS_LONG scl;
        scl = (FS_SHORT) SWAPWINC(p); /* as 2.14 */

        trans[0] = trans[3] = (scl << 2);
        trans[1] = trans[2] = 0;
        key->localTIsIdentity = 0;
    }
    else if (f & WE_HAVE_AN_X_AND_Y_SCALE)
    {
        FS_LONG scl;
        scl = (FS_SHORT) SWAPWINC(p); /* as 2.14 */
        trans[0] = (scl << 2);
        scl = (FS_SHORT) SWAPWINC(p); /* as 2.14 */
        trans[3] = (scl << 2);
        trans[1] = trans[2] = 0;
        key->localTIsIdentity = 0;
    }
    else if (f & WE_HAVE_A_TWO_BY_TWO)
    {
        FS_LONG scl;
        scl = (FS_SHORT) SWAPWINC(p); /* as 2.14 */
        trans[0] = (scl << 2);
        scl = (FS_SHORT) SWAPWINC(p); /* as 2.14 */
        trans[1] = (scl << 2);
        scl = (FS_SHORT) SWAPWINC(p); /* as 2.14 */
        trans[2] = (scl << 2);
        scl = (FS_SHORT) SWAPWINC(p); /* as 2.14 */
        trans[3] = (scl << 2);
        key->localTIsIdentity = 0;

    }
    else /* no scale */
    {
        trans[0] = trans[3] = FIXED_ONE;
        trans[1] = trans[2] = 0;
    }

    *more_comps = (f & MORE_COMPONENTS) == MORE_COMPONENTS;

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("read_comp_parms_ttf => %d 0x%04x %d %d [%.3f, %.3f, %.3f, %.3f] %d\n",
               *index, *flags, *arg1, *arg2,
               trans[0] / 65536.0, trans[1] / 65536.0,
               trans[2] / 65536.0, trans[3] / 65536.0,
               *more_comps));
#endif

    /*  just update the byte pointer */
    bio->array = (FS_BYTE *)p;
    return bio;
}
/****************************************************************/
#ifdef FS_CCC
BITIO *read_comp_parms_ccc(_DS_
                           fsg_SplineKey *key,
                           BITIO *bio,
                           FS_USHORT *index,
                           FS_USHORT *flags,
                           FS_SHORT *arg1,
                           FS_SHORT *arg2,
                           FS_FIXED *trans,
                           FS_BOOLEAN *more_comps)
{
    FS_USHORT flag;
    int arg12Words;
    FS_SHORT getBits, shiftBits;
    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);
    getBits = ttf->ccc_info.numBits_SF;
    shiftBits = 16 - getBits;

    /* ARG_1_AND_2_ARE_WORDS */
    arg12Words = MTX_BITIO_ReadValue(bio, 1);
    flag = (FS_USHORT)arg12Words;

    /*** ??? CAN'T USE ANCHOR POINTS ??? ***/
    flag |= ARGS_ARE_XY_VALUES;

    /* ROUND_XY_TO_GRID */
    if (STATE.cur_lfnt->fontflags & FONTFLAG_CCC)
    {
        if (MTX_BITIO_ReadValue(bio, 1))
            flag |= ROUND_XY_TO_GRID;
    }

    /* bits 3-7 ... come from 1-3 bits in file */

    /* 0    WE_HAVE_AN_X_AND_Y_SCALE
    *  10    no scale
    *  110  WE_HAVE_A_SCALE
    *  111  WE_HAVE_A_TWO_BY_TWO
    *
    * for a 1.5 MB file, SansMTUniK_v848a.stf
    * this saves a WHOPPING 47863 bits versus
    * using 2 bits for each possibility. That
    * is 0.38% ... hardly seems worthwhile.
    *
    */


    if (MTX_BITIO_ReadValue(bio, 1) == 0)
    {
        flag |= WE_HAVE_AN_X_AND_Y_SCALE;
    }
    else
    {
        if (MTX_BITIO_ReadValue(bio, 1) == 1)
        {
            if (MTX_BITIO_ReadValue(bio, 1) == 0)
                flag |= WE_HAVE_A_SCALE;
            else
                flag |= WE_HAVE_A_TWO_BY_TWO;
        }
        /* else no-scale */
    }

    /*  NB : uses bio->high as number of components remaining */
    if (--bio->high != 0)    /* more components ? */
    {
        flag |= MORE_COMPONENTS;
        *more_comps = 1;
    }
    else
        *more_comps = 0;


    if (STATE.cur_lfnt->fontflags & FONTFLAG_DDD)
    {
        int v = ttf->maxp->numGlyphs;
        FS_SHORT numBits = 0;

        while (v)
        {
            numBits++;
            v >>= 1;
        }

        *index = (FS_USHORT) MTX_BITIO_ReadValue(bio, numBits);
    }
    else
        *index = (FS_USHORT) MTX_BITIO_ReadValue(bio, 16);

    if (arg12Words == 1)
    {
        *arg1     = (FS_SHORT)MTX_BITIO_ReadValue(bio, 16);
        *arg2     = (FS_SHORT)MTX_BITIO_ReadValue(bio, 16);
    }
    else
    {
        *arg1 = (FS_SHORT)(FS_TINY)MTX_BITIO_ReadValue(bio, 8);
        *arg2 = (FS_SHORT)(FS_TINY)MTX_BITIO_ReadValue(bio, 8);
    }

    if (flag & WE_HAVE_A_TWO_BY_TWO)
    {
        FS_FIXED temp;

        temp = (FS_USHORT)MTX_BITIO_ReadValue(bio, getBits); /* unsigned 10 bits */
        temp <<= shiftBits;           /* move bit-9 into sign position */
        temp = (FS_SHORT)temp;        /* convert to signed short */
        trans[0] = (temp << 2);       /* convert to 2.16 */

        temp = (FS_USHORT)MTX_BITIO_ReadValue(bio, getBits);
        temp <<= shiftBits;
        temp = (FS_SHORT)temp;
        trans[1] = (temp << 2);

        temp = (FS_USHORT)MTX_BITIO_ReadValue(bio, getBits);
        temp <<= shiftBits;
        temp = (FS_SHORT)temp;
        trans[2] = (temp << 2);

        temp = (FS_USHORT)MTX_BITIO_ReadValue(bio, getBits);
        temp <<= shiftBits;
        temp = (FS_SHORT)temp;
        trans[3] = (temp << 2);
        key->localTIsIdentity = 0;

    }
    else
    {
        FS_FIXED temp;

        trans[1] = trans[2] = 0;
        if (flag &  WE_HAVE_AN_X_AND_Y_SCALE)
        {
            temp = (FS_USHORT)MTX_BITIO_ReadValue(bio, getBits);
            temp <<= shiftBits;
            temp = (FS_SHORT)temp;
            temp <<= 2;
            trans[0] = temp;

            temp = (FS_USHORT)MTX_BITIO_ReadValue(bio, getBits);
            temp <<= shiftBits;
            temp = (FS_SHORT)temp;
            temp <<= 2;
            trans[3] = temp;
            key->localTIsIdentity = 0;
        }
        else if (flag & WE_HAVE_A_SCALE)
        {
            temp = (FS_USHORT)MTX_BITIO_ReadValue(bio, getBits);
            temp <<= shiftBits;
            temp = (FS_SHORT)temp;
            temp <<= 2;
            trans[0] = trans[3] = temp;
            key->localTIsIdentity = 0;
        }
    }

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("read_comp_parms_ccc => %d 0x%04x %d %d [%.3f, %.3f, %.3f, %.3f] %d\n",
               *index, *flags, *arg1, *arg2,
               trans[0] / 65536.0, trans[1] / 65536.0,
               trans[2] / 65536.0, trans[3] / 65536.0,
               *more_comps));
#endif


    /* version8 CCC store USE_MY_METRICS bit */

    if (STATE.cur_lfnt->fontflags & FONTFLAG_CCC)
    {
        if (ttf->ccc_info.cccVersion8)
        {
            if (MTX_BITIO_ReadValue(bio, 1) == 1)
                flag |= USE_MY_METRICS;
        }
    }

    if (*more_comps == 0)
    {
        if (MTX_BITIO_ReadValue(bio, 1) == 1)
            flag |= WE_HAVE_INSTRUCTIONS;
        bio->index += 7;
    }

    *flags = flag;

    return bio;
}
#endif /* FS_CCC */

/****************************************************************/
BITIO *read_comp_parms(_DS_
                       fsg_SplineKey *key,
                       BITIO *bio,
                       FS_USHORT *index,
                       FS_USHORT *flags,
                       FS_SHORT *arg1,
                       FS_SHORT *arg2,
                       FS_FIXED *trans,
                       FS_BOOLEAN *more_comps)
{

#ifdef FS_CCC
    if ((STATE.cur_lfnt->fontflags & (FONTFLAG_CCC | FONTFLAG_DDD)))
        return read_comp_parms_ccc(_PS_ key, bio, index, flags, arg1, arg2, trans, more_comps);
    else
#else
    FS_state_ptr = FS_state_ptr; /* avoid compiler warning */
#endif /* FS_CCC */
        return read_comp_parms_ttf(key, bio, index, flags, arg1, arg2, trans, more_comps);
}
/****************************************************************/
FS_VOID update_sp_ep(int comps, fnt_ElementType *elementPtr)
{

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("update_sp_ep(%d)\n", comps));
#endif

    if (comps > 0)
    {
        int n, nc;
        FS_USHORT *sp = elementPtr->sp;
        FS_USHORT *ep = elementPtr->ep;

        /* number of contours in this composite */
        nc = elementPtr->nc;

        /* number of contours in previous component */
        n = elementPtr->ep[-1] + 1;
        while (nc--)
        {
            *sp++ += (FS_USHORT)n;
            *ep++ += (FS_USHORT)n;
        }
    }
#ifdef NC_DEBUG
    dump_elementPtr(elementPtr, "after 'update_sp_ep'");
#endif
}

/****************************************************************/
/* advance elementPtr to next 'empty slot */
static FS_VOID advance_elementPtr(fsg_SplineKey *key, fnt_ElementType *p)
{
    int nc, np;

    nc = p->nc;
    np = nc ? (1 + p->ep[nc - 1] - p->sp[0]) : 0;
    fsg_IncrementElement(key, 1, np, nc);
    p->nc = 0;
}

/****************************************************************/
/* this is the TTF version - a separate function exists for CCC */
/* this version prefers speed over memory usage                 */
#ifdef FS_SPEED_OVER_MEMORY
static FS_VOID get_glyph_metrics_ttf(_DS_ fsg_SplineKey *key, FS_USHORT gIndex, fsg_Metrics *metrics)
{
    TTF *ttf = (FS_VOID *)((LFNT *)((SFNT *)(key->sfnt)->lfnt)->fnt);
    FS_USHORT num = (TTF_HHEA *)(ttf->hhea)->numberOf_LongHorMetrics;
    HMTX* hmtx;

    if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
    {
        hmtx = (HMTX *)(FS_ULONG *)(ttf->hmtx);
    }
    else
    {
        hmtx = (HMTX *)ttf->hmtx;
    }

    if (gIndex < num)
    {
        hmtx += gIndex;  /* point to hmtx entry for gIndex */

        metrics->aw  = SWAPW(hmtx->advanceWidth);
        metrics->lsb = SWAPW(hmtx->leftSideBearing);
    }
    else    /* gIndex >= num */
    {
        FS_BYTE *p;
        FS_SHORT lsb;

        /* last HMTX element ... get it's AW */
        hmtx += (num - 1);

        metrics->aw  = SWAPW(hmtx->advanceWidth);

        /* now get lsb from array of lsb's that follow */
        p = (FS_BYTE *)hmtx + sizeof(HMTX) + (gIndex - num) * 2;
        lsb = (FS_SHORT) * p;
        metrics->lsb = SWAPW(lsb);
    }

    if (ttf->vmtx)
    {
        VMTX *vmtx;

        if  (!ttf->memptr || (FS_VOID *)(ttf->decomp) )
        {
            vmtx = (VMTX *)(FS_ULONG *)(ttf->vmtx);
        }
        else
        {
            vmtx = (VMTX *)ttf->vmtx;
        }

        num = (TTF_VHEA *)(ttf->vhea)->numberOf_LongVerMetrics;
        if (gIndex < num)
        {
            vmtx += gIndex;  /* point to vmtx entry for gIndex */

            metrics->ah  = SWAPW(vmtx->advanceHeight);
            metrics->tsb = SWAPW(vmtx->topSideBearing);
        }
        else    /* gIndex >= num */
        {
            FS_BYTE *p;
            FS_SHORT tsb;

            /* last VMTX element ... get it's AH */
            vmtx += (num - 1);

            metrics->ah  = SWAPW(vmtx->advanceHeight);

            p = (FS_BYTE *)vmtx + sizeof(VMTX) + (gIndex - num) * 2;
            tsb = (FS_SHORT) * p;
            metrics->tsb = SWAPW(tsb);
        }
    }
    else
    {
        TTF_OS2  *pos2 = (TTF_OS2 *)ttf->os2;
        FS_SHORT ascender = 0;
        FS_SHORT descender = 0;
        if (pos2)
        {
            ascender = pos2->sTypoAscender;
            descender = pos2->sTypoDescender;
        }
        else
        {
            ascender = phhea->yAscender;
            descender = phhea->yDescender;
        }
        metrics->ah = ascender - descender;
        metrics->tsb = (TTF_HEAD *)(ttf->head)->yMax - ascender;
    }

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("get_glyph_metrics_ttf: aw=%d lsb=%d\n", metrics->aw, metrics->lsb));
#endif

}
#else /* this version convserves memory at the cost of speed */
static FS_VOID get_glyph_metrics_ttf(_DS_ fsg_SplineKey *key, FS_USHORT gIndex, fsg_Metrics *metrics)
{
    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);
    TTF_HHEA  *phhea = (TTF_HHEA *)ttf->hhea;
    FS_USHORT num = phhea->numberOf_LongHorMetrics;
    FS_ULONG offset = ttf->hmtx_offset;
    HMTX hmtx;
    FS_SHORT lsb;

    hmtx.advanceWidth = 0;    /* without this line, coverity reports uninit_use */
    hmtx.leftSideBearing = 0; /* without this line, coverity reports uninit_use */

    if (gIndex < num)
    {
        /* compute the byte offset of the correct HMTX element and read it */
        offset += gIndex * sizeof(HMTX);
        ttf_read_buf(_PS_ ttf, offset, sizeof(HMTX), (FS_BYTE *)&hmtx);
        if (STATE.error)
            return;

        metrics->aw  = SWAPW(hmtx.advanceWidth);
        metrics->lsb = SWAPW(hmtx.leftSideBearing);
    }
    else    /* gIndex >= num */
    {
        /* byte offset of last HMTX element ... get it's AW */
        lsb = 0;    /* without this line, coverity reports uninit_use */

        offset += (num - 1) * sizeof(HMTX);
        ttf_read_buf(_PS_ ttf, offset, sizeof(HMTX), (FS_BYTE *)&hmtx);

        if (STATE.error)
            return;
        metrics->aw  = SWAPW(hmtx.advanceWidth);

        offset += sizeof(HMTX) + (gIndex - num) * 2;
        ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&lsb);

        if (STATE.error)
            return;
        metrics->lsb = SWAPW(lsb);
    }

    offset = ttf->vmtx_offset;
    if (offset)
    {
        VMTX vmtx;
        FS_SHORT tsb;

        vmtx.advanceHeight = 0;  /* without this line, coverity reports uninit_use */
        vmtx.topSideBearing = 0; /* without this line, coverity reports uninit_use */

        num = ttf->vhea->numberOf_LongVerMetrics;
        if (gIndex < num)
        {
            /* compute the byte offset of the correct VMTX element and read it */
            offset += gIndex * sizeof(VMTX);
            ttf_read_buf(_PS_ ttf, offset, sizeof(VMTX), (FS_BYTE *)&vmtx);
            if (STATE.error)
                return;

            metrics->ah  = SWAPW(vmtx.advanceHeight);
            metrics->tsb = SWAPW(vmtx.topSideBearing);
        }
        else    /* gIndex >= num */
        {
            tsb = 0; /* without this line, coverity reports uninit_use */

            /* byte offset of last VMTX element ... get it's AH */
            offset += (num - 1) * sizeof(VMTX);
            ttf_read_buf(_PS_ ttf, offset, sizeof(VMTX), (FS_BYTE *)&vmtx);

            if (STATE.error)
                return;
            metrics->ah  = SWAPW(vmtx.advanceHeight);

            offset += sizeof(VMTX) + (gIndex - num) * 2;
            ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&tsb);

            if (STATE.error)
                return;
            metrics->tsb = SWAPW(tsb);
        }
    }
    else
    {
        TTF_OS2  *pos2 = (TTF_OS2 *)ttf->os2;
        FS_SHORT ascender = 0;
        FS_SHORT descender = 0;
        if (pos2)
        {
            ascender = pos2->sTypoAscender;
            descender = pos2->sTypoDescender;
        }
        else
        {
            ascender = phhea->yAscender;
            descender = phhea->yDescender;
        }
        metrics->ah = ascender - descender;
        metrics->tsb = ttf->head->yMax - ascender;
    }

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("get_glyph_metrics_ttf: aw=%d lsb=%d\n", metrics->aw, metrics->lsb));
#endif

}
#endif /* FS_SPEED_OVER_MEMORY */

/****************************************************************/
#ifdef FS_CCC
static FS_VOID get_glyph_metrics_ccc(_DS_ fsg_SplineKey *key, FS_USHORT gIndex, fsg_Metrics *metrics)
{
    FS_BOOLEAN cccVersion7, cccVersion8;
    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);
    TTF_HHEA  *phhea = (TTF_HHEA *)ttf->hhea;
    FS_USHORT num = phhea->numberOf_LongHorMetrics;
    FS_ULONG offset;
    BITIO bio;
    FS_BYTE MemTemp[5];
    int n_aw, n_lsb, bit_pos, first, last, bytes;
    int n_ah, n_tsb;


    cccVersion7 = ttf->ccc_info.cccVersion7;
    cccVersion8 = ttf->ccc_info.cccVersion8;
    if ( cccVersion7 || cccVersion8 )
    {
        offset = ttf->hmtx_offset + 8;
        n_aw = ttf->ccc_info.numBits_AW;
    }
    else
    {
        offset = ttf->hmtx_offset + 4;
        n_aw = 16;
    }

    n_lsb = ttf->ccc_info.numBits_LSB;

    if (gIndex < num)
    {
        bit_pos = gIndex * (n_aw + n_lsb);
        first = bit_pos >> 3;
        last = (bit_pos + n_aw + n_lsb - 1) >> 3;
        bytes = 1 + last - first;
        ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
        if (STATE.error)
            return;

        bio.array = &MemTemp[0];
        bio.index = bit_pos & 7;

        metrics->aw = (FS_USHORT) MTX_BITIO_ReadValue(&bio, n_aw);
        if ( cccVersion7 || cccVersion8 )
            metrics->aw += ttf->ccc_info.minVal_AW;

        metrics->lsb = (FS_SHORT) MTX_BITIO_ReadValue(&bio, ttf->ccc_info.numBits_LSB);
        metrics->lsb += ttf->ccc_info.minVal_LSB;
    }
    else    /* gIndex >= num */
    {
        /* get aw from last [aw,lsb] entry */
        bit_pos = (num - 1) * (n_aw + n_lsb);
        first = bit_pos >> 3;
        last = (bit_pos + n_aw - 1) >> 3;
        bytes = 1 + last - first;
        ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
        if (STATE.error)
            return;
        bio.array = MemTemp;
        bio.index = bit_pos & 7;
        metrics->aw = (FS_USHORT) MTX_BITIO_ReadValue(&bio, n_aw);
        if ( cccVersion7 || cccVersion8 )
            metrics->aw += ttf->ccc_info.minVal_AW;

        /* get the proper lsb[] entry */
        bit_pos += n_aw + n_lsb;
        bit_pos += (gIndex - num) * n_lsb;
        first = bit_pos >> 3;
        last = (bit_pos + n_lsb - 1) >> 3;
        bytes = 1 + last - first;
        ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
        if (STATE.error)
            return;
        bio.index = bit_pos & 7;
        metrics->lsb = (FS_SHORT)MTX_BITIO_ReadValue(&bio, n_lsb);
        metrics->lsb += ttf->ccc_info.minVal_LSB;
    }

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("get_glyph_metrics_ccc: aw=%d lsb=%d\n", metrics->aw, metrics->lsb));
#endif
    /* VMTX table */
    offset = ttf->vmtx_offset;
    if (offset)
    {
        num = ttf->vhea->numberOf_LongVerMetrics;
        if ( cccVersion7 || cccVersion8 )
        {
            offset += 8;
            n_ah = ttf->ccc_info.numBits_AH;
            n_tsb = ttf->ccc_info.numBits_TSB;
            if (gIndex < num)
            {
                bit_pos = gIndex * (n_ah + n_tsb);
                first = bit_pos >> 3;
                last = (bit_pos + n_ah + n_tsb - 1) >> 3;
                bytes = 1 + last - first;
                ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
                if (STATE.error)
                    return;
                bio.array = &MemTemp[0];
                bio.index = bit_pos & 7;
                metrics->ah = (FS_USHORT) MTX_BITIO_ReadValue(&bio, n_ah);
                metrics->ah += ttf->ccc_info.minVal_AH;
                metrics->tsb = (FS_SHORT) MTX_BITIO_ReadValue(&bio, ttf->ccc_info.numBits_TSB);
                metrics->tsb += ttf->ccc_info.minVal_TSB;
            }
            else    /* gIndex >= num */
            {
                /* get aw from last [aw,lsb] entry */
                bit_pos = (num - 1) * (n_ah + n_tsb);
                first = bit_pos >> 3;
                last = (bit_pos + n_ah - 1) >> 3;
                bytes = 1 + last - first;
                ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
                if (STATE.error)
                    return;
                bio.array = MemTemp;
                bio.index = bit_pos & 7;
                metrics->ah = (FS_USHORT) MTX_BITIO_ReadValue(&bio, n_ah);
                metrics->ah += ttf->ccc_info.minVal_AH;

                /* get the proper lsb[] entry */
                bit_pos += n_ah + n_tsb;
                bit_pos += (gIndex - num) * n_tsb;
                first = bit_pos >> 3;
                last = (bit_pos + n_tsb - 1) >> 3;
                bytes = 1 + last - first;
                ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
                if (STATE.error)
                    return;
                bio.index = bit_pos & 7;
                metrics->tsb = (FS_SHORT)MTX_BITIO_ReadValue(&bio, n_tsb);
                metrics->tsb += ttf->ccc_info.minVal_LSB;
            }

        }
        else
        {
            VMTX vmtx;
            FS_SHORT tsb;

            vmtx.advanceHeight = 0;  /* without this line, coverity reports uninit_use */
            vmtx.topSideBearing = 0; /* without this line, coverity reports uninit_use */

            num = ttf->vhea->numberOf_LongVerMetrics;
            if (gIndex < num)
            {
                /* compute the byte offset of the correct VMTX element and read it */
                offset += gIndex * sizeof(VMTX);
                ttf_read_buf(_PS_ ttf, offset, sizeof(VMTX), (FS_BYTE *)&vmtx);
                if (STATE.error)
                    return;

                metrics->ah  = SWAPW(vmtx.advanceHeight);
                metrics->tsb = SWAPW(vmtx.topSideBearing);
            }
            else    /* gIndex >= num */
            {
                tsb = 0; /* without this line, coverity reports uninit_use */

                /* byte offset of last VMTX element ... get it's AH */
                offset += (num - 1) * sizeof(VMTX);
                ttf_read_buf(_PS_ ttf, offset, sizeof(VMTX), (FS_BYTE *)&vmtx);

                if (STATE.error)
                    return;
                metrics->ah  = SWAPW(vmtx.advanceHeight);

                offset += sizeof(VMTX) + (gIndex - num) * 2;
                ttf_read_buf(_PS_ ttf, offset, 2, (FS_BYTE *)&tsb);

                if (STATE.error)
                    return;
                metrics->tsb = SWAPW(tsb);
            }
        }
    }
    else
    {
        TTF_OS2  *pos2 = (TTF_OS2 *)ttf->os2;
        FS_SHORT ascender = 0;
        FS_SHORT descender = 0;
        if (pos2)
        {
            ascender = pos2->sTypoAscender;
            descender = pos2->sTypoDescender;
        }
        else
        {
            ascender = phhea->yAscender;
            descender = phhea->yDescender;
        }
        metrics->ah = ascender - descender;
        metrics->tsb = ttf->head->yMax - ascender;
    }
}

static FS_VOID get_glyph_metrics_ddd(_DS_ fsg_SplineKey *key, FS_USHORT gIndex, fsg_Metrics *metrics)
{

    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);
    FS_ULONG offset;
    FS_ULONG tableoffset;
    FS_ULONG tableindex;
    BITIO bio;
    FS_BYTE MemTemp[5];
    FS_BYTE b;
    FS_USHORT idx;
    int n_aw,  bit_pos, first, last, bytes;
    TTF_HHEA  *phhea = (TTF_HHEA *)ttf->hhea;
    FS_USHORT num = phhea->numberOf_LongHorMetrics;


    tableoffset = ttf->hmtx_offset + 2;

    offset = tableoffset + ttf->ccc_info.minVal_LSB;

    n_aw = ttf->ccc_info.numBits_AW;

    idx = gIndex;

    if ( gIndex >= num )
        idx = num - 1;


    bit_pos = idx * (n_aw);
    first = bit_pos >> 3;
    last = (bit_pos + n_aw - 1) >> 3;
    bytes = 1 + last - first;
    ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
    if (STATE.error)
        return;

    bio.array = &MemTemp[0];
    bio.index = bit_pos & 7;

    tableindex = (FS_USHORT) MTX_BITIO_ReadValue(&bio, n_aw);

    ttf_read_buf(_PS_ ttf, tableoffset + tableindex, bytes, &b);

    metrics->aw = b;
    if (metrics->aw == 255)
        metrics->aw++;


#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("get_glyph_metrics_ccc: aw=%d lsb=%d\n", metrics->aw, metrics->lsb));
#endif
    /* VMTX table */
    offset = ttf->vmtx_offset;
    if (offset)
    {
        tableoffset = ttf->hmtx_offset + 2;

        offset = tableoffset + ttf->ccc_info.minVal_TSB;

        n_aw = ttf->ccc_info.numBits_AH;


        bit_pos = gIndex * (n_aw);
        first = bit_pos >> 3;
        last = (bit_pos + n_aw - 1) >> 3;
        bytes = 1 + last - first;
        ttf_read_buf(_PS_ ttf, offset + first, bytes, MemTemp);
        if (STATE.error)
            return;

        bio.array = &MemTemp[0];
        bio.index = bit_pos & 7;

        tableindex = (FS_USHORT) MTX_BITIO_ReadValue(&bio, n_aw);

        ttf_read_buf(_PS_ ttf, tableoffset + tableindex, bytes, &b);

        metrics->ah = b;
        if (metrics->ah == 255)
            metrics->ah++;
    }
    else
    {
        TTF_OS2  *pos2 = (TTF_OS2 *)ttf->os2;
        FS_SHORT ascender = 0;
        FS_SHORT descender = 0;
        if (pos2)
        {
            ascender = pos2->sTypoAscender;
            descender = pos2->sTypoDescender;
        }
        else
        {
            ascender = phhea->yAscender;
            descender = phhea->yDescender;
        }
        metrics->ah = ascender - descender;
        metrics->tsb = ttf->head->yMax - ascender;
    }
}
#endif
/****************************************************************/
FS_VOID get_glyph_metrics(_DS_ fsg_SplineKey *key, FS_USHORT gIndex, fsg_Metrics *metrics)
{
#ifdef FS_CCC
    if (STATE.cur_lfnt->fontflags & FONTFLAG_CCC)
        get_glyph_metrics_ccc(_PS_ key, gIndex, metrics);
    else if (STATE.cur_lfnt->fontflags & FONTFLAG_DDD)
        get_glyph_metrics_ddd(_PS_ key, gIndex, metrics);
    else
#endif
        get_glyph_metrics_ttf(_PS_ key, gIndex, metrics);
}

/****************************************************************/
/* returns scaled, unhinted advance width                       */
FS_VOID get_glyph_advance(_DS_ fsg_SplineKey *key, FS_USHORT gIndex,
                          FS_FIXED *dx, FS_FIXED *dy)
{
    fsg_Metrics metrics;
    F26DOT6 x = 0, y = 0, oox, ooy;

    *dx = *dy = 0;

    get_glyph_metrics(_PS_ key, gIndex, &metrics);
    if (STATE.error) return;

    if (STATE.flags & FLAGS_VERTICAL_ON)
    {
        oox = 0;
        ooy = metrics.ah;
    }
    else
    {
        oox = metrics.aw;
        ooy = 0;
    }
    scale_piece(key, &x, &y, &oox, &ooy, 1);
    *dx = ((x + 32)&~63) << 10; /* round F26DOT6 and convert to FS_FIXED */
    *dy = ((y + 32)&~63) << 10;
}

/****************************************************************/
/* this is the TTF version -- write a seperate function for CCC */
BITIO *get_charData_ttf(BITIO *bio, sfnt_PackedSplineFormat *charData)
{
    FS_SHORT *p = (FS_SHORT *)bio->array;

    /* basic info */
    charData->numberContours = (FS_SHORT)SWAPWINC(p);
    charData->bbox.xMin = (FS_SHORT)SWAPWINC(p);
    charData->bbox.yMin = (FS_SHORT)SWAPWINC(p);    /* not used */
    charData->bbox.xMax = (FS_SHORT)SWAPWINC(p);    /* not used */
    charData->bbox.yMax = (FS_SHORT)SWAPWINC(p);    /* not used */

    /* ? simple character */
    if (charData->numberContours >= 0)
    {
        charData->endPoints = p;
        p += charData->numberContours;
        charData->numberInstructions = (FS_SHORT)SWAPWINC(p);
        charData->instructions = (FS_BYTE*)p;
        charData->flags = (FS_BYTE*)p + charData->numberInstructions;
    }
    else
    {
        charData->endPoints = 0;
        charData->numberInstructions = 0;
        charData->instructions = 0;
        charData->flags = 0;
    }

    bio->array = (FS_BYTE *)p;
    return bio;
}
/****************************************************************/
#ifdef FS_CCC
BITIO *get_charData_ccc(_DS_ BITIO *bio, sfnt_PackedSplineFormat *charData)
{
    int comp;
    TTF *ttf = (FS_VOID *)((LFNT *)(STATE.cur_sfnt->lfnt)->fnt);

    /* !!! really MUST have the xMIN !!! */
    charData->bbox.xMin = 0;
    charData->bbox.yMin = 0;
    charData->bbox.xMax = 0;
    charData->bbox.yMax = 0;

    comp = MTX_BITIO_ReadValue(bio, 1);
    if (comp)
    {
        /* compound character */
        charData->numberContours = -1;

        /* read number of components */
        bio->high = MTX_BITIO_ReadValue(bio, ttf->ccc_info.numBits_GLYF);
    }
    else
    {
        /* simple char -- read number of contours */
        FS_SHORT *p;

        bio->index--; /*  rewind the bit ... better be on a byte boundary */

        p = ((FS_SHORT *)bio->array + (bio->index >> 3));
        charData->numberContours = SWAPW(*p);
        p++;
        charData->endPoints = p;
        p += charData->numberContours;
        charData->numberInstructions = (FS_SHORT)SWAPW(*p);
        p++;
        charData->instructions = (FS_BYTE*)p;
        charData->flags = (FS_BYTE*)p + charData->numberInstructions;
    }
    return bio;
}
#endif /* FS_CCC */
/****************************************************************/
BITIO *get_charData(_DS_ BITIO *bio, sfnt_PackedSplineFormat *charData)
{
#ifdef FS_CCC
    if ( (STATE.cur_lfnt->fontflags & FONTFLAG_CCC) ||
            (STATE.cur_lfnt->fontflags & FONTFLAG_DDD) )
        return get_charData_ccc(_PS_ bio, charData);
    else
#else
    FS_state_ptr = FS_state_ptr; /* avoid compiler warning */
#endif
        return get_charData_ttf(bio, charData);
}
/****************************************************************/
FS_VOID apply_transMatrix(fsg_SplineKey *key, F26DOT6 *x, F26DOT6 *y, int np, transMatrix *trans)
{
    FS_FIXED *m = &trans->transform[0][0];
    FS_FIXED m0, m1, m3, m4;

    if ( np == 0 )
        np += PHANTOMCOUNT;

    m0 = m[0];
    m1 = m[1];
    m3 = m[3];
    m4 = m[4];

    {
        /* we've already accounted for interpScalar X/Y */
        /* when scaling the character ... don't apply them again */
        fnt_GlobalGraphicStateType* globalGS = GLOBALGSTATE(key);
        if ( m0 == globalGS->interpScalarX &&
             m4 == globalGS->interpScalarY &&
             m1 == 0 && m3 == 0)
        {
            /* very common case */
            m0 = m4 = FIXED_ONE;
            /* m1 = m3 = 0; */
        }
        else
        {
            m0 = FixDiv(m0, globalGS->interpScalarX);
            m1 = FixDiv(m1, globalGS->interpScalarX);
            m3 = FixDiv(m3, globalGS->interpScalarY);
            m4 = FixDiv(m4, globalGS->interpScalarY);
        }
    }

    /* diagonal ? */
    if (m1 == 0 && m3 == 0)
    {
        int n;
        if (m0 != FIXED_ONE)
        {
            n = np;
            while (n--)
            {
                *x = scale_26_6_by_fix(*x, m0);
                x++;
            }
        }
        if (m4 != FIXED_ONE)
        {
            n = np;
            while (n--)
            {
                *y = scale_26_6_by_fix(*y, m4);
                y++;
            }
        }
    }
    else /* generic 2x2 */
    {
        F26DOT6 tx, ty;

        while (np--)
        {
            tx = *x;
            ty = *y;
            *x++ = scale_26_6_by_fix(tx, m0) + scale_26_6_by_fix(ty, m3);
            *y++ = scale_26_6_by_fix(tx, m1) + scale_26_6_by_fix(ty, m4);
        }
    }
}

/****************************************************************/
/* try and combine scanControl in a rational way ....
* But, I don't think there always IS a rational way to combine them.
* for instance: component A says DONT EVER perform dropout control
* and component B says do it if under 75 lpm, or stretched, or rotated
* What's the rational combination?
*/
static FS_VOID update_scanControl(fsg_SplineKey *key)
{
    fnt_GlobalGraphicStateType* globalGS = GLOBALGSTATE(key);
    FS_ULONG new_sc = globalGS->localParBlock.scanControl;
    FS_ULONG old_sc;
    FS_ULONG new_ppm;
    FS_ULONG old_ppm;

    if (new_sc == 0) return;

    /* separate flags and thresholds */
    new_ppm = (new_sc & 0x00FF);
    new_sc &= 0xFFFFFF00;
    old_sc = key->scanControl;
    old_ppm = (old_sc & 0x00FF);
    old_sc &= 0xFFFFFF00;

    /* OR the flags ... use the larger threshsold */
    new_sc |= old_sc;
    new_ppm = MAX(old_ppm, new_ppm);
    key->scanControl = (new_sc | new_ppm);
}


/****************************************************************/
#ifdef FS_HINTS
int which_script(_DS_ int index, FS_ULONG *unicode)
{
    FS_USHORT platform = STATE.platform;
    FS_USHORT encoding = STATE.encoding;
    FS_ULONG flags = STATE.flags;

    int script = 0;

    FSS_set_cmap(_PS_ 3, 1);
    if (STATE.error)
    {
        FSS_set_cmap(_PS_ 3, 10);
    }

    if (STATE.error)
    {
        FSS_set_cmap(_PS_ platform, encoding);
        return 0xFFFF;
    }

    FS_set_flags(_PS_ FLAGS_CMAP_ON);
    if (STATE.error)
    {
        FSS_set_cmap(_PS_ platform, encoding);
        STATE.flags = flags;
        return 0xFFFF;
    }

    *unicode = inverse_map_char(_PS_ index + STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->start_index, 
                                &(STATE.cur_typeset.tfntarray[STATE.cur_font]));

    FSS_set_cmap(_PS_ platform, encoding);
    STATE.flags = flags;

    /* un-encoded OR private use area (plane 0) */
    if (*unicode == 0xFFFF || (0xE000 <= *unicode && *unicode <= 0xF8FF))
    {
        /* if the font supports only one of our scripts (other than Latin/Greek/Cyrillic)
           then the unencoded/private_use characters should be of that script ... Right ??? */
        LFNT *lfnt = STATE.cur_lfnt;
        TTF *ttf = (TTF *)(lfnt->fnt);
        FS_ULONG r1 = ttf->os2->ulUnicodeRange1;
        FS_ULONG r2 = ttf->os2->ulUnicodeRange2;
        FS_ULONG r3 = ttf->os2->ulUnicodeRange3;
        FS_ULONG b1, b2, b3;
        int count = 0;

        /* what are these bit constants that follow ??
           http://www.microsoft.com/typography/OTSpec/os2.htm#ur */

        if (r1 & (1 << 11))
        {
            count++;
            script = SCRIPT_HEBREW;
        }

        b1 = (1 << 13);
        b2 = ((FS_ULONG)1 << (63 - 32));
        b3 = (1 << (67 - 64));
        if ((r1 & b1) | (r2 & b2) | (r3 & b3))
        {
            count++;
            script = SCRIPT_ARABIC;
        }

        b2 = (1 << (52 - 32)) | (1 << (56 - 32));
        if (r2 & b2)
        {
            count++;
            script = SCRIPT_HANGUL;
        }

        b2 = (1 << (48 - 32)) | (1 << (54 - 32)) | (1 << (55 - 32)) | (1 << (59 - 32)) | (1 << (61 - 32));
        if (r2 & b2)
        {
            count++;
            script = SCRIPT_CJK;
        }

        if (r1 & (1 << 15))
        {
            count++;
            script = SCRIPT_DEVANAGARI;
        }

        if (r1 & (1 << 16))
        {
            count++;
            script = SCRIPT_BENGALI;
        }

        if (r1 & (1 << 17))
        {
            count++;
            script = SCRIPT_GURMUHKI;
        }

        if (r1 & (1 << 18))
        {
            count++;
            script = SCRIPT_GUJARATI;
        }
        if (r1 & (1 << 19))
        {
            count++;
            script = SCRIPT_ORIYA;
        }

        if (r1 & (1 << 20))
        {
            count++;
            script = SCRIPT_TAMIL;
        }

        if (r1 & (1 << 21))
        {
            count++;
            script = SCRIPT_TELUGU;
        }

        if (r1 & (1 << 22))
        {
            count++;
            script = SCRIPT_KANNADA;
        }

        if (r1 & (1 << 23))
        {
            count++;
            script = SCRIPT_MALAYALAM;
        }

        if (r1 & (1 << 24))
        {
            count++;
            script = SCRIPT_THAI;
        }

        if (r3 & (1 << (73 - 64)))
        {
            count++;
            script = SCRIPT_SINHALA;
        }

        if (r3 & (1 << (80 - 64)))
        {
            count++;
            script = SCRIPT_KHMER;
        }


        /* add others here */

        /* exactly one script return it ... else return 'other' */
        if (count != 1)
            script = SCRIPT_OTHER;
    }

    /* encoded character ranges */
    else if (0x0020 <= *unicode && *unicode <= 0x007F)
        script = SCRIPT_LATIN;    /* basic latin */
    else if (0x0080 <= *unicode && *unicode <= 0x00FF)
        script = SCRIPT_LATIN;    /* latin-1 Supplement */
    else if (0x0100 <= *unicode && *unicode <= 0x017F)
        script = SCRIPT_LATIN;    /* latin Ext A */
    else if (0x0180 <= *unicode && *unicode <= 0x024F)
        script = SCRIPT_LATIN;    /* latin Ext B */
    else if (0x0250 <= *unicode && *unicode <= 0x02AF)
        script = SCRIPT_LATIN;    /* IPA Ext */
    else if (0x1D00 <= *unicode && *unicode <= 0x1D7F)
        script = SCRIPT_LATIN;    /* Phonetic Ext */
    else if (0x1E00 <= *unicode && *unicode <= 0x1EFF)
        script = SCRIPT_LATIN;    /* Latin Ext Additional */
    else if (0x2C60 <= *unicode && *unicode <= 0x2C7F)
        script = SCRIPT_LATIN;    /* Latin Ext C */
    else if (0xA720 <= *unicode && *unicode <= 0xA7FF)
        script = SCRIPT_LATIN;    /* Latin Ext D */
    else if (0xFB00 <= *unicode && *unicode <= 0xFB06)
        script = SCRIPT_LATIN;    /* latin presentation forms */
    else if (0x0370 <= *unicode && *unicode <= 0x03FF)
        script = SCRIPT_GREEK;    /* greek */
    else if (0x1F00 <= *unicode && *unicode <= 0x1FFF)
        script = SCRIPT_GREEK;    /* greek */
    else if (0x2C80 <= *unicode && *unicode <= 0x2CFF)
        script = SCRIPT_GREEK;    /* coptic */
    else if (0x0400 <= *unicode && *unicode <= 0x04FF)
        script = SCRIPT_CYRILLIC;    /* cyrillic */
    else if (0x0500 <= *unicode && *unicode <= 0x052F)
        script = SCRIPT_CYRILLIC;    /* cyrillic Supplement */
    else if (0x2DE0 <= *unicode && *unicode <= 0x2DFF)
        script = SCRIPT_CYRILLIC;    /* cyrillic Extended A */
    else if (0xA640 <= *unicode && *unicode <= 0xA69F)
        script = SCRIPT_CYRILLIC;    /* cyrillic Extended B */
    else if (0x0590 <= *unicode && *unicode <= 0x05FF)
        script = SCRIPT_HEBREW;    /* hebrew */
    else if (0xFB1D <= *unicode && *unicode <= 0xFB4F)
        script = SCRIPT_HEBREW;    /* hebrew */
    else if (0x0600 <= *unicode && *unicode <= 0x06FF)
        script = SCRIPT_ARABIC;    /* arabic */
    else if (0x0750 <= *unicode && *unicode <= 0x077F)
        script = SCRIPT_ARABIC;    /* arabic Supplement */
    else if (0xFB50 <= *unicode && *unicode <= 0xFDFF)
        script = SCRIPT_ARABIC;    /* arabic */
    else if (0xFE70 <= *unicode && *unicode <= 0xFEFF)
        script = SCRIPT_ARABIC;    /* arabic */
    else if (0xAC00 <= *unicode && *unicode <= 0xD7AF)
        script = SCRIPT_HANGUL;    /* hangul */
    else if (0x2E80 <= *unicode && *unicode <= 0x2FD5)
        script = SCRIPT_CJK;    /* CJK radicals */
    else if (0x31C0 <= *unicode && *unicode <= 0x31EF)
        script = SCRIPT_CJK;    /* CJK strokes */
    else if (0x3400 <= *unicode && *unicode <= 0x4DBF)
        script = SCRIPT_CJK;    /* CJK Ext A */
    else if (0x4E00 <= *unicode && *unicode <= 0x9FFF)
        script = SCRIPT_CJK;    /* CJK unified ideographs */
    else if (0xF900 <= *unicode && *unicode <= 0xF9FF)
        script = SCRIPT_CJK;    /* CJK compatibilty */
    else if (0x0900 <= *unicode && *unicode <= 0x097F)
        script = SCRIPT_DEVANAGARI;
    else if (0x0980 <= *unicode && *unicode <= 0x09FF)
        script = SCRIPT_BENGALI;
    else if (0x0A00 <= *unicode && *unicode <= 0x0A7F)
        script = SCRIPT_GURMUHKI;
    else if (0x0A80 <= *unicode && *unicode <= 0x0AFF)
        script = SCRIPT_GUJARATI;
    else if (0x0B00 <= *unicode && *unicode <= 0x0B7F)
        script = SCRIPT_ORIYA;
    else if (0x0B80 <= *unicode && *unicode <= 0x0BFF)
        script = SCRIPT_TAMIL;
    else if (0x0C00 <= *unicode && *unicode <= 0x0C7F)
        script = SCRIPT_TELUGU;
    else if (0x0C80 <= *unicode && *unicode <= 0x0CFF)
        script = SCRIPT_KANNADA;
    else if (0x0D00 <= *unicode && *unicode <= 0x0D7F)
        script = SCRIPT_MALAYALAM;
    else if (0x0D80 <= *unicode && *unicode <= 0x0DFF)
        script = SCRIPT_SINHALA;
    else if (0x0E00 <= *unicode && *unicode <= 0x0E7F)
        script = SCRIPT_THAI;
    else if (0x1780 <= *unicode && *unicode <= 0x17FF)
        script = SCRIPT_KHMER;

    return script;
}
#endif /* FS_HINTS */



/****************************************************************/

FS_SHORT get_comp(_DS_
                  fsg_SplineKey *key,
                  FS_USHORT index,
                  FS_FIXED *c_trans,    /* current accumulated local transformations */
                  FS_USHORT flags,
                  FS_SHORT arg1,
                  FS_SHORT arg2,
                  FS_FIXED *p_trans,    /* prior local transformation */
                  int useHints,
                  fsg_Metrics *metrics)
{
    FS_SHORT *glyph_ptr = 0;
    TTF *ttf = (FS_VOID *)(STATE.cur_lfnt->fnt);
    FS_ULONG length, offset, nc=0, np;
    fnt_ElementType *elementPtr;
    sfnt_PackedSplineFormat charData = {0};
    FS_SHORT result = 0;
#ifdef NC_DEBUG
    FS_SHORT base_contours = key->num_contours;
#endif

#ifdef FS_HINTS
    FS_BYTE RTGA[3] = {0xb0, 11, 0x7f};
#endif /* FS_HINTS */

    fsg_Metrics l_metrics = {0};

    BITIO bitio;            /*  overkill for TTF's, but */
    BITIO *bio = &bitio;    /*  needed for CCC's */

#ifdef NC_DEBUG
    {
        double z = 65536.0;

        indent_fn();
        FS_PRINTF(("get_comp(%d, [%.3f, %.3f, %.3f, %.3f], 0x%04x, %d, %d, [%.3f, %.3f, %.3f, %.3f])\n",
                   index, c_trans[0] / z, c_trans[1] / z, c_trans[2] / z, c_trans[3] / z, flags,
                   arg1, arg2, p_trans[0] / z, p_trans[1] / z, p_trans[2] / z, p_trans[3] / z));

    }
#endif /* NC_DEBUG */

    /* current elementPtr points to last component we got (if any) */
    elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);

    /* validate glyph index */
    if (index >= ttf->maxp->numGlyphs)
        index = 0;

    /* update the "is really an outline" bit -- assuming that composite chars */
    /* are either ALL outlines or ALL stiks ... combinations will not work !!! */
#ifdef FS_STIK
    if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) && (!STATE.outl_char) &&
        (STATE.cur_lfnt->fontflags & (FONTFLAG_CCC | FONTFLAG_DDD | FONTFLAG_ACT3)))
        STATE.outl_char |= get_nstk_bit(_PS_ index);    /* NotSTiK character */
#endif

    /* locate glyph data */
    get_glyph_parms(_PS_ ttf, index, &offset, &length);
    if (STATE.error)
    {
        FSS_free(_PS_ glyph_ptr);
        return result;
    }

    /* get glyph metrics */
    if (STATE.level == 0)
    {
        l_metrics.ah = metrics->ah;
        l_metrics.aw = metrics->aw;
        l_metrics.lsb = metrics->lsb;
        l_metrics.tsb = metrics->tsb;
    }
    else
        get_glyph_metrics(_PS_ key, index, &l_metrics);

    /* if there IS glyphData, read it and extract charData */
    /* else charData is all 0's which is the correct thing */
    if (length > 0)
    {

        /* read glyph data */
#ifdef FS_MEM_DBG
        STATE.memdbgid = "bio->array";
#endif
        glyph_ptr = (FS_SHORT *)ttf_read(_PS_ ttf, ttf->glyf_offset + offset, length);
        if (STATE.error)
        {
            FSS_free(_PS_ glyph_ptr);
            return result;
        }

        bio->array = (FS_BYTE *)glyph_ptr;
        bio->index = 0;

        /* get number_contours, bbox,  etc.. */
        bio = get_charData(_PS_ bio, &charData);

#ifdef FS_CCC
        /*** THIS IS A KLUDGE ... ***MUST*** HAVE THE XMIN IN THE CCC DATA ***/
        /*** because it is NOT ALWAYS the case that xMin == lsb ***/
        charData.bbox.xMin = l_metrics.lsb;
#endif
    }

    /* simple character */
    if (charData.numberContours >= 0)
    {
        FS_BYTE *ptr;
        FS_LONG num;

        advance_elementPtr(key, elementPtr);

        STATE.error = sfnt_UnfoldCurve(key, &charData, &num, &ptr, length );
        if (STATE.error)
        {
            FSS_free(_PS_ glyph_ptr);
            return result;
        }

#ifdef NC_DEBUG
        indent_fn();
        FS_PRINTF(("sfnt_UnfoldCurve added %d contours and %d points\n", elementPtr->nc, elementPtr->ep[elementPtr->nc - 1] + 1));
#endif

#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "after sfntUnfoldCurve");
#endif

        /* initialize phantom points */
        init_phantom_points(elementPtr, &charData.bbox, &l_metrics);

        /* scale coordinates with combined transformation */
        scale_char(key, elementPtr);

        /* copy <x,y> => <ox,oy> */
        fsg_CopyElementBackwards(elementPtr);


#ifdef EDIT_MODE
        if (STATE.level == 0 && instructions)
        {
            num = instructionLength;
            ptr = instructions;
        }
#endif

#ifdef FS_HINTS
        /* control whether RTGAH is used */
        if ( useHints &&                                                /* hinting enabled */
             (STATE.flags  & FLAGS_GRAYSCALE) &&                        /* grayscale */
             !(STATE.cur_lfnt->fontflags & FONTFLAG_STIK) &&            /* outline (not stroke) */
             !(STATE.cur_sfnt->rtgah_suitable == RTGAH_NOPE))           /* it's not unsuitable */
        {
            if (STATE.flags & FLAGS_FORCE_AUTOHINT )
            {
                /* we are forcing the issue... but only for simple outlines */
                if (STATE.level == 0)
                {
                    key->globalGS.rtgah_data.script = which_script(_PS_ index, &(key->globalGS.rtgah_data.unicode));
                    ptr = RTGA;
                    num = 3;
                    STATE.any_hints |= OUTL_FLAGS_RTGAH;
                }
                else
                {
                    /* we are forcing the issue, but not for components */
                    ptr = 0;
                    num = 0;
                }
            }
            else if ( num == 0 && STATE.level == 0 &&                   /* simple un-hinted glyph */
                      !(STATE.flags  & FLAGS_AUTOHINT_OFF))               /* RTGAH has not been turned off */
            {
#ifdef FS_EDGE_HINTS
                if ( key->globalGS.maz_data.gridFitType == ADF_GRID_FIT_NONE )
#endif
                {
                    key->globalGS.rtgah_data.script = which_script(_PS_ index, &(key->globalGS.rtgah_data.unicode));
                    ptr = RTGA;
                    num = 3;
                    STATE.any_hints |= OUTL_FLAGS_RTGAH;
                }
            }
        }
#endif /* FS_HINTS */

#ifdef FS_STIK
        if ((charData.flags) && (STATE.cur_lfnt->fontflags & FONTFLAG_STIK) &&
            !(STATE.cur_lfnt->fontflags & (FONTFLAG_CCC | FONTFLAG_DDD | FONTFLAG_ACT3)) && (!STATE.outl_char))
            STATE.outl_char |= charData.flags[0] & OUTLINE_CHAR;    /* NotSTiK character */
#endif
        /* update first byte of flag */
        elementPtr->f[0] = STATE.outl_char; /* qwu 03-01-06 */

        if (STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON )
        {
            /* apply any transformation */
            apply_transformation(key, elementPtr, c_trans);

            /* apply any hints */
            do_hints(_PS_ key, elementPtr, num, ptr, 0, useHints, &l_metrics);

            /* then offsets */
            apply_offsets(_PS_ key, key->num_comps, flags, arg1, arg2, p_trans, elementPtr);
        }
        else
        {
            /* apply any hints */
            do_hints(_PS_ key, elementPtr, num, ptr, 0, useHints, &l_metrics);

#ifdef FS_HINTS
            if (ptr == RTGA)
            {
                /* the autohinter may set the FONTFLAG_NO_AUTOHINT bit
                 * if it becomes confused by a character.   */
                STATE.cur_lfnt->fontflags = key->globalGS.rtgah_data.fontflags;
            }
#endif /* FS_HINTS */

            /* apply any transformation */
            apply_transformation(key, elementPtr, c_trans);

            /* then offsets */
            apply_offsets(_PS_ key, key->num_comps, flags, arg1, arg2, p_trans, elementPtr);
        }

        /* update total components, contours and points */
        key->num_comps++;
        key->num_contours += elementPtr->nc;
        nc = elementPtr->nc;
        np = nc ? (1 + elementPtr->ep[nc - 1] - elementPtr->sp[0]) : 0;
        key->num_points += np;

        result = elementPtr->nc;
    }
    else if (charData.numberContours == -1)
    {
        FS_SHORT base_nc;
        FS_LONG base_np;
        FS_BOOLEAN more_comps;
        FS_BOOLEAN composite_instructions = 0;    /* SWP_NEW */
        FS_SHORT num = 0;
        FS_USHORT l_index;
        FS_USHORT l_flags;
        FS_SHORT l_arg1;
        FS_SHORT l_arg2;
        FS_FIXED l_trans[4];
        FS_FIXED n_trans[4];

#ifdef FS_STATS
        size_composite_char += length;
#endif

        /* Save where this composite started so we can return
        * the proper elementPtr when the composite is complete.
        * We need to get there to apply composite level hints
        * and it is (afterall) the result of this function call.
        */
        base_nc = key->num_contours;
        base_np = key->num_points;
#ifdef FS_STATS
        num_composite_chars++;
#endif

#ifdef NC_DEBUG
        nested++;
        indent_fn();
        FS_PRINTF(("*** composite: base_nc=%d base_np=%d\n", base_nc, base_np));
#endif

#ifdef NC_DEBUG
        indent += 4;
#endif
        if (STATE.level == 0)
        {
            comp_phantom_points(key, elementPtr, &charData.bbox, metrics);
            update_bearings(elementPtr, key, base_np, 0, useHints);
            set_metrics(metrics, elementPtr, base_np);
            num = 0;
        }

        /*useMyMetricsOn = false;*/    /* qwu 07-03-06 */
        do
        {
            bio = read_comp_parms(_PS_ key, bio, &l_index, &l_flags, &l_arg1, &l_arg2, l_trans, &more_comps);

            if (l_flags & WE_HAVE_INSTRUCTIONS)
                composite_instructions |= 1;

            if (l_flags & ANY_SCALE)
                /* compose the current local scale with prior accumulated scale */
                compose_trans(n_trans, l_trans, c_trans);
            else
                /* it's the identy matrix ... just copy c_trans */
                SYS_MEMCPY(n_trans, c_trans, 16);

            /* if multiple components in a composite set the */
            /* USE_MY_METRICS bit, the last one will win ... */
            metrics->useMyMetrics = (l_flags & USE_MY_METRICS) ? 1 : 0;
            /*if ( metrics->useMyMetrics )
                useMyMetricsOn = true; */    /* qwu 07-03-06 */

            /* get the component (which itself may be a composite ) */
            STATE.level++;
            if (l_index != key->glyphIndex) /* avoids infinite recursion */
            {
                nc = get_comp(_PS_ key, l_index, n_trans, l_flags, l_arg1, l_arg2, c_trans, useHints, metrics);
                if (STATE.error)
                {
                    FSS_free(_PS_ glyph_ptr);
                    return result;
                }

                if (STATE.level == 1 && l_flags & USE_MY_METRICS)
                {
                    /* only for level 1 components */
                    int nce = elementPtr->nc;
                    int npe = nce ? elementPtr->ep[elementPtr->nc - 1] + 1 : 0;
                    set_metrics(metrics, elementPtr, npe);
                    metrics->useMyMetrics = 1;
                }
            }

            STATE.level--;

            /* if we added any contours, update SP[] and EP[] wrt previous component */
            if (nc)
                update_sp_ep(num++, elementPtr);

        }
        while (more_comps);


#ifdef NC_DEBUG
        indent -= 4;
        indent_fn();
        FS_PRINTF(("resetting to base via fsg_IncrementElement(%d,%d)\n", base_np, base_nc));
#endif

        /* reset to starting point of this composite [base_nc, base_np] */
        fsg_SetUpElement(_PS_ key, (FS_LONG)GLYPHELEMENT);
        fsg_IncrementElement(key, (FS_LONG)GLYPHELEMENT, base_np, base_nc);

        /* set total number of contours in this composite */
        elementPtr->nc = key->num_contours - base_nc;

        /* restore the composites phantom points */
        /* comp_phantom_points(key,elementPtr,c_trans,&charData.bbox,metrics); */ /* qwu 07-03-06 */

#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "after reset elementPtr to base_nc base_np");
#endif

        /* restore the composites phantom points */
        /*if (useMyMetricsOn) */ /* qwu 07-03-06 */
        {
            int nce = elementPtr->nc;
            np = nce ? elementPtr->ep[elementPtr->nc - 1] + 1 : 0;
            get_metrics(elementPtr, metrics, np);
        }
#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "after restoring phantom points");
#endif

        /* use prior transformation for offsets as x/y coords */
        apply_offsets(_PS_ key, key->num_comps, flags, arg1, arg2, p_trans, elementPtr);

        /* copy <x,y> -> <ox,oy> */
        fsg_CopyElementBackwards(elementPtr);

        /* do the composite hints if any */
        num = 0;

#ifdef FS_HINTS
        if (useHints && STATE.level == 0 &&                     /* hints enabled, simple glyph */
            (STATE.flags  & FLAGS_GRAYSCALE)  &&                /* grayscale */
            !(STATE.cur_lfnt->fontflags & FONTFLAG_STIK) &&     /* outline (not stroke) */
            !(STATE.cur_sfnt->rtgah_suitable == RTGAH_NOPE))    /* it's not unsuitable */
        {
            if ((STATE.flags & FLAGS_FORCE_AUTOHINT) ||         /* we are forcing OR */
                ((composite_instructions == 0) &&               /* compound gray character */
                !(STATE.any_hints & OUTL_FLAGS_ANYHINTS) &&     /* without hints */
#ifdef FS_EDGE_HINTS
                (key->globalGS.maz_data.gridFitType == ADF_GRID_FIT_NONE) &&
#endif
                !(STATE.flags & FLAGS_AUTOHINT_OFF)))           /* and RTGAH has not been turned off */
                composite_instructions = 255;    /* a (255) can only come from RUN_TIME_GRAY_AUTOHINT */
        }
#endif /* FS_HINTS */

#ifdef EDIT_MODE
        if (STATE.level == 0 && instructions)
        {
            do_hints(_PS_ key, elementPtr, instructionLength, instructions, 1, useHints, metrics);
        }
        else
#endif

        if (composite_instructions)
        {
            FS_BYTE *p;
#ifdef FS_HINTS
            if (composite_instructions == 255)
            {
                key->globalGS.rtgah_data.script = which_script(_PS_ index, &(key->globalGS.rtgah_data.unicode));
                p = RTGA;
                num = 3;
                STATE.any_hints |= OUTL_FLAGS_RTGAH;
            }
            else
#endif /* FS_HINTS */
            {

                if (bio->index & 7)
                {
                    /*** NOTE: bio->index is always non-zero on a non CCC font ***/
                    /* FS_PRINTF(("*** composite level hints are NOT on a byte boundary \n")); */
                }
                p = bio->array + (bio->index >> 3);
                num = GET_xWORD(p);
                p += 2;
            }

            /* FS_PRINTF(("composite hints: num = %d\n",num)); */
            /* if (num == 0)    FS_PRINTF(("*** supposed to be composite hints, but num==0\n")); */


            do_hints(_PS_ key, elementPtr, num, p, 1, useHints, metrics);

#ifdef FS_STATS
                num_composite_hints += num;
#endif
        }
        else
        {
            do_hints(_PS_ key, elementPtr, 0, NULL, 1, useHints, metrics);
        }

#ifdef NC_DEBUG
        indent_fn();
        FS_PRINTF(("base__contours=%d key->num_contour=%d\n",
                   base_contours, key->num_contours));
#endif

        /* return number of (new) contours in this (possibly nested) composite */
        result = elementPtr->nc;

#ifdef NC_DEBUG
        dump_elementPtr(elementPtr, "after compound 'get_comp'");
#endif

    }
    else
    {
        STATE.error = ERR_BAD_GLYF_FORMAT;
        result = 0;
        FSS_free(_PS_ glyph_ptr);
        return result;
    }

    /* update key->scanControl */
    update_scanControl(key);

    if (glyph_ptr)
        FSS_free(_PS_ glyph_ptr);
    return result;
}
/****************************************************************/
FS_LONG fsg_GridFit(_DS_ fsg_SplineKey *key, FS_BOOLEAN useHints)
{
    FS_USHORT flags = ARGS_ARE_XY_VALUES;
    FS_SHORT arg1 = 0;
    FS_SHORT arg2 = 0;
    FS_FIXED p_trans[4] = {FIXED_ONE, 0, 0, FIXED_ONE};
    FS_FIXED trans[4] = {FIXED_ONE, 0, 0, FIXED_ONE};
    FS_SHORT nc;
    fsg_Metrics metrics = {0};
    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);
    fnt_ElementType *elementPtr;
    FS_LONG np;
    FS_USHORT gIndex = (FS_USHORT)key->glyphIndex;

    /* start out  empty */
    key->num_comps = 0;
    key->num_contours = 0;
    key->num_points = 0;

    STATE.level = 0;
    STATE.any_hints = 0;
    STATE.outl_char = 0;    /* it's not an outline char in a stik font ... so far */

#ifdef NC_DEBUG
    nested = 0;
#endif



    /* set elementPtr to 0-th slot */
    fsg_SetUpElement(_PS_ key, (FS_LONG)TWILIGHTZONE );
    fsg_SetUpElement(_PS_ key, (FS_LONG)GLYPHELEMENT);
    elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);

#ifdef FS_HINTS
    if (key->sfnt != STATE.server->workspace_sfnt)
    {
        /* workspace no longer corresponds to key sfnt, re-run preprogram */
        STATE.server->workspace_sfnt = key->sfnt;

        STATE.error = fsg_RunPreProgram(_PS_ key);
        if (STATE.error)
            return STATE.error;
    }
#endif

    /* if user said NO_HINTS .. no hints, else ask the font */
    useHints = !(STATE.flags & FLAGS_HINTS_OFF);
    if (useHints)
        if (globalGS->localParBlock.instructControl & NOGRIDFITFLAG )
            useHints = 0;

#ifdef NC_DEBUG
    indent_fn();
    FS_PRINTF(("get_char(%d)\n", gIndex));
#endif

    /* get the glyph metrics, then store AW and LSB in key */
    get_glyph_metrics(_PS_ key, gIndex, &metrics);

    key->localTIsIdentity = 1;

    /* get the character as is with no offsets or scale */
    nc = get_comp(_PS_ key, gIndex, p_trans, flags, arg1, arg2, trans, (int)useHints, &metrics);
    if (STATE.error)
        return STATE.error;
    np = nc ? (1 + PHANTOMCOUNT + elementPtr->ep[nc - 1] - elementPtr->sp[0]) : 0;

#ifdef NC_DEBUG
    {
        fnt_ElementType *elementPtr;
        elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
        dump_elementPtr(elementPtr, "after 'before apply_transMatrix'");
    }
#endif

    /* apply the base transformation to get rotation/oblique/stretch */
    apply_transMatrix(key, elementPtr->x, elementPtr->y, (int)np, &key->currentTMatrix);

#ifdef NC_DEBUG
    {
        fnt_ElementType *elementPtr;
        elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);
        dump_elementPtr(elementPtr, "after 'apply_transMatrix'");
        FS_PRINTF((" ***nested=%ld\n", nested));
    }
#endif

    return STATE.error;
}
/****************************************************************/
#ifdef FS_STIK
/* is the glyph <index> marked as a REAL outline in the 'nstk' table */
FS_BYTE get_nstk_bit(_DS_ FS_USHORT index)
{
    FS_USHORT *list = (FS_USHORT *)(((TTF *)(FS_VOID *)((LFNT *)(STATE.cur_sfnt->lfnt)->fnt))->nstk);
    FS_LONG lo, mid, hi;

    /* ? no list */
    if (list == 0 || list[0] == 0)
        return 0;

    /* ? outside of list */
    lo = 1;
    hi = list[0];
    if (index < list[lo] || index > list[hi])
        return 0;

    /* binary search list */
    while (lo <= hi)
    {
        mid = (lo + hi) / 2;
        if (index < list[mid])
            hi = mid - 1;
        else if (index > list[mid])
            lo = mid + 1;
        else
            return OUTLINE_CHAR; /* found it: it's really an outline */
    }
    /* not found ... just a plain stick */
    return 0;
}
#endif /* FS_STIK */

/****************************************************************/
/******** usable stuff from old fsglue.c ************************/
/****************************************************************/
/* delete memory allocated for key (or globalGS) */
FS_VOID delete_key(_DS_ FS_VOID *p)
{
    fsg_SplineKey *key = (fsg_SplineKey *)p;
    /*    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key); */

    /* these are passed in from our TTF struct ... delete them there
    * FSS_free(_PS_ globalGS->pgmList[PREPROGRAM]);
    * FSS_free(_PS_ globalGS->pgmList[FONTPROGRAM]);
    *
    *    don't delete this yet ... it IS the key ... we still need it
    *    FSS_free(key->memoryBases[KEY_PTR_BASE]);
    */

    if (key->memoryBases[VOID_FUNC_PTR_BASE]) FSS_free(_PS_ (FS_BYTE *)(key->memoryBases[VOID_FUNC_PTR_BASE]));
    FSS_free(_PS_ (FS_BYTE *)(key->memoryBases[PRIVATE_FONT_SPACE_BASE]));

    /* now */
    FSS_free(_PS_ key);
}


/**********************************************************************************/
static FS_ULONG fsg_SetOffsetPtr(fsg_OffsetInfo *offsetPtr, FS_ULONG workSpacePos, FS_ULONG maxPoints, FS_ULONG maxContours)
{
    FS_ULONG      ArraySize;

    offsetPtr->interpreterFlagsOffset = workSpacePos;

    workSpacePos += maxPoints * sizeof (FS_BYTE);

    FS_ALIGN (workSpacePos);

    offsetPtr->startPointOffset = workSpacePos;
    ArraySize = maxContours * sizeof (FS_SHORT);

    FS_ALIGN(ArraySize);
    workSpacePos += ArraySize;
    offsetPtr->endPointOffset = workSpacePos;
    workSpacePos += ArraySize;

    offsetPtr->oldXOffset = workSpacePos;
    ArraySize = maxPoints * sizeof (F26DOT6);

    FS_ALIGN(ArraySize);
    workSpacePos += ArraySize;
    offsetPtr->oldYOffset    = workSpacePos;
    workSpacePos += ArraySize;
    offsetPtr->scaledXOffset = workSpacePos;
    workSpacePos += ArraySize;
    offsetPtr->scaledYOffset = workSpacePos;
    workSpacePos += ArraySize;
    offsetPtr->newXOffset    = workSpacePos;
    workSpacePos += ArraySize;
    offsetPtr->newYOffset    = workSpacePos;
    workSpacePos += ArraySize;

    offsetPtr->onCurveOffset = workSpacePos;
    workSpacePos += maxPoints * sizeof (FS_BYTE);
    FS_ALIGN (workSpacePos);

    return workSpacePos;
}

#ifdef FS_EDGE_HINTS

static FS_ULONG fsg_SetOffsetPtrforMAZ(fsg_OffsetInfo *offsetPtr, FS_ULONG workSpacePos, FS_ULONG maxPoints, FS_ULONG maxContours)
{
    FS_ULONG originalPos = workSpacePos;

    offsetPtr->MAZOutlineContoursOffset = workSpacePos;
    workSpacePos += maxContours * sizeof(MAZOutlineContour);
    FS_ALIGN (workSpacePos);

    offsetPtr->MAZOutlinePointsOffset = workSpacePos;
    workSpacePos += maxPoints * sizeof(MAZOutlinePoint);
    FS_ALIGN (workSpacePos);

    offsetPtr->MAZHSegsOffset = workSpacePos;
    workSpacePos += (maxPoints >> 1) * MAX(sizeof(MAZOutlineSegment), sizeof(MAZStrokeSegment)); /*lint !e506 Suppress 'Warning -- Constant value Boolean' */
    FS_ALIGN (workSpacePos);

    offsetPtr->MAZVSegsOffset = workSpacePos;
    workSpacePos += (maxPoints >> 1) * MAX(sizeof(MAZOutlineSegment), sizeof(MAZStrokeSegment)); /*lint !e506 Suppress 'Warning -- Constant value Boolean' */
    FS_ALIGN (workSpacePos);

    offsetPtr->MAZOutlineContourListOffset = workSpacePos;
    workSpacePos += maxContours * sizeof(MAZOutlineContour *);
    FS_ALIGN (workSpacePos);

    offsetPtr->MAZOutlineSegPairsOffset = workSpacePos;
    workSpacePos += (maxPoints >> 1) * sizeof(MAZOutlineSegPair);
    FS_ALIGN (workSpacePos);

    return (workSpacePos - originalPos);  /* return size difference */
}
#endif /* FS_EDGE_HINTS */

static FS_ULONG fsg_SetOffsetPtrforRTGAH(fsg_OffsetInfo *offsetPtr, FS_ULONG workSpacePos,
                                         FS_ULONG maxPoints, FS_ULONG maxContours)
{
    FS_ULONG originalPos = workSpacePos;

    (void)maxContours; /* unused. Note that MAX_CONTOUS is used below. This is done
                          because some fonts have an incorrect max contours, and using
                          this incorrect value could cause a crash. */

    offsetPtr->RTGAHPiecesOffset = workSpacePos;
    workSpacePos += MAX_PIECES * sizeof(PIECE);
    FS_ALIGN (workSpacePos);

    offsetPtr->RTGAHStrokesOffset = workSpacePos;
    workSpacePos += MAX_STROKES * sizeof(STROKE);
    FS_ALIGN (workSpacePos);

    offsetPtr->RTGAHStrokesPlistOffset  = workSpacePos;
    /* workSpacePos += 2 * maxPoints * sizeof(PLIST);  */
    /* I had forgotten about combined strokes */
    /* this can re-use PLISTS multiple times */
    /* I do not know the proper high limit */
    workSpacePos += 4 * maxPoints * sizeof(PLIST);
    FS_ALIGN (workSpacePos);

    offsetPtr->RTGAHEdgesOffset = workSpacePos;
    workSpacePos += 2 * MAX_STROKES * sizeof(EDGE);
    FS_ALIGN (workSpacePos);

    offsetPtr->RTGAHXminOffset = workSpacePos;
    workSpacePos += MAX_CONTOURS * sizeof(int);
    FS_ALIGN (workSpacePos);

    offsetPtr->RTGAHYminOffset = workSpacePos;
    workSpacePos += MAX_CONTOURS * sizeof(int);
    FS_ALIGN (workSpacePos);

    offsetPtr->RTGAHXmaxOffset = workSpacePos;
    workSpacePos += MAX_CONTOURS * sizeof(int);
    FS_ALIGN (workSpacePos);

    offsetPtr->RTGAHYmaxOffset = workSpacePos;
    workSpacePos += MAX_CONTOURS * sizeof(int);

    FS_ALIGN (workSpacePos);

    return (workSpacePos - originalPos);  /* return size difference */
}

/**********************************************************************************/
/*
 * fsg_WorkSpaceSetOffsets : This stuff changes with each glyph
 *
 * Computes the workspace size and sets the offsets into it.
 *
 */
FS_ULONG fsg_WorkSpaceSetOffsets(fsg_SplineKey *key)
{
    FS_ULONG workSpacePos, maxPoints, maxContours;
    FS_ULONG MAZworkspace = 0, RTGAHworkspace;
    TTF_MAXP *maxp = (TTF_MAXP *)(key->maxp);    /* already swapped */
#ifdef FS_EDGE_HINTS
    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);
#endif

    key->elementInfoRec.stackBaseOffset = workSpacePos = 0;
    workSpacePos += (maxp->maxStackElements) * sizeof (F26DOT6);

    /* ELEMENT 0 */

    workSpacePos = fsg_SetOffsetPtr(&(key->elementInfoRec.offsets[TWILIGHTZONE]),
                                    workSpacePos,
                                    (FS_ULONG)maxp->maxTwilightPoints,
                                    MAX_TWILIGHT_CONTOURS);
#ifdef FS_EDGE_HINTS
    /* should not need a MAZ workspace for the twilight zone, but it is not too large */
    /* and avoids a crash with some Edge fonts. No workspace is needed for RTGAH.     */
    if (ttf->adfh)
    {
        MAZworkspace = fsg_SetOffsetPtrforMAZ(&(key->elementInfoRec.offsets[TWILIGHTZONE]),
                                              workSpacePos,
                                              (FS_ULONG)maxp->maxTwilightPoints,
                                              MAX_TWILIGHT_CONTOURS);
        workSpacePos += MAZworkspace;
    }
#endif

    /* ELEMENT 1 */

    maxPoints = maxp->maxPoints;
    if (maxPoints < maxp->maxCompositePoints)
        maxPoints = maxp->maxCompositePoints;
    maxPoints += PHANTOMCOUNT;

    maxContours = maxp->maxContours;
    if (maxContours < maxp->maxCompositeContours)
        maxContours = maxp->maxCompositeContours;

    workSpacePos = fsg_SetOffsetPtr(&(key->elementInfoRec.offsets[GLYPHELEMENT]),
                                    workSpacePos,
                                    maxPoints,
                                    maxContours);

    /* MAZ and RTGAH share a workspace area in the glyph element */
    /* workspace area is big enough to accommodate either        */
    /*lint -e587 -e685  Warning 587,685: Warning -- predicate > always evaluates to false  */

#ifdef FS_EDGE_HINTS
    if (ttf->adfh)
        MAZworkspace = fsg_SetOffsetPtrforMAZ(&(key->elementInfoRec.offsets[GLYPHELEMENT]),
                                              workSpacePos,
                                              maxPoints,
                                              maxContours);
#endif

    RTGAHworkspace = fsg_SetOffsetPtrforRTGAH(&(key->elementInfoRec.offsets[GLYPHELEMENT]),
                                              workSpacePos,
                                              maxPoints,
                                              maxContours);

    workSpacePos += MAX(MAZworkspace, RTGAHworkspace);
    /*lint +e587 +e685  Warning 587,685: Warning -- predicate > always evaluates to false  */

    return workSpacePos;
}

/**********************************************************************************/
/*
 *    fsg_CopyElementBackwards
 */
FS_VOID fsg_CopyElementBackwards(fnt_ElementType *elementPtr)
{
    FS_USHORT nc = elementPtr->nc;
    int np;

    if (nc == 0) return;
    np = 1 + PHANTOMCOUNT + elementPtr->ep[nc - 1] - elementPtr->sp[0];
    SYS_MEMCPY(elementPtr->ox, elementPtr->x, np * 4);
    SYS_MEMCPY(elementPtr->oy, elementPtr->y, np * 4);
    SYS_MEMSET(elementPtr->f, 0, np);
}

/**********************************************************************************/
/*
 *    Inverse scale the hinted, scaled points back into the FRU domain.
 */
#ifdef FS_HINTS
static FS_VOID fsg_SnapShotOutline( fsg_SplineKey* key, fnt_ElementType* elem, FS_LONG numPts )
{
    FS_LONG bigUpem = (FS_LONG)key->emResolution << 10;
    FS_FIXED scalar = key->interpScalar;
    F26DOT6 *oox = elem->oox;
    F26DOT6 *x = elem->x;
    F26DOT6 *ooy = elem->ooy;
    F26DOT6 *y = elem->y;
    FS_FIXED scl;

    scl = FixDiv(bigUpem, scalar);
    while (numPts-- >= 0)
    {
        *oox++ = scale_26_6_by_fix( *x++, scl);
        *ooy++ = scale_26_6_by_fix( *y++, scl);
    }
}
#endif /* FS_HINTS */

/**********************************************************************************/
static FS_FIXED fsg_MaxAbs(FS_FIXED a, FS_FIXED b)
{
    if (a < 0) a = -a;
    if (b < 0) b = -b;
    return a > b ? a : b;
}


/**********************************************************************************/
/*
 * Returns true if we have the identity matrix.
 */
static FS_BYTE fsg_Identity( transMatrix *matrix )
{
    FS_FIXED* m = &matrix->transform[0][0];
    FS_FIXED hi, lo;

    /* exactly ? */
    if (m[0] == FIXED_ONE && m[1] == 0 && m[2] == 0 &&
            m[3] == 0 && m[4] == FIXED_ONE && m[5] == 0)
        return 1;

    /* close enough ? */
    hi = FIXED_ONE + ALMOSTZERO;
    lo = FIXED_ONE - ALMOSTZERO;
    if ((m[0] >= lo && m[0] <= hi) &&
        (m[1] >= -ALMOSTZERO && m[1] <= ALMOSTZERO) &&
        (m[2] >= -ALMOSTZERO && m[2] <= ALMOSTZERO) &&
        (m[3] >= -ALMOSTZERO && m[3] <= ALMOSTZERO) &&
        (m[4] >= lo && m[4] <= hi) &&
        (m[5] >= -ALMOSTZERO && m[5] <= ALMOSTZERO))
        return 1;

    return 0;

}

/**********************************************************************************/

static FS_BOOLEAN fsg_Non90Degrees( transMatrix *matrix )
{
    return ( (matrix->transform[0][0] || matrix->transform[1][1]) &&
             (matrix->transform[1][0] || matrix->transform[0][1]) );
}


/*lint -e441 Warning -- for clause irregularity: loop variable 'shift' not found in 2nd for expression */
/**********************************************************************************/
/*
 *    counts number of low bits that are zero
 *    -- or --
 *    returns bit number of first ON bit
 */
static int fsg_CountLowZeros( FS_ULONG n )
{
    int shift;
    FS_ULONG one = 1;

    if (n == 0) return 32;

    for (shift = 0; !( n & one ); shift++)
        n >>= 1;
    return shift;
}
/*lint +e441 Warning -- for clause irregularity: loop variable 'shift' not found in 2nd for expression */

/**********************************************************************************/
/*
 * fsg_GetShift
 * return 2log of n if n is a power of 2 otherwise -1;
 */
static int fsg_GetShift( FS_ULONG n )
{
    if (ISNOTPOWEROF2(n) || !n)
        return -1;
    else
        return fsg_CountLowZeros( n );
}

/**********************************************************************************/
/*
 * fsg_InitInterpreterTrans                <3>
 *
 * Computes [xy]TransMultiplier in global graphic state from matrix
 * It leaves the actual matrix alone
 * It then sets these key flags appropriately
 *        identityTransformation        1 == no need to run points through matrix
 *        imageState                     pixelsPerEm
 *        imageState                    Rotate flag if glyph is rotated
 *        imageState                    Stretch flag if glyph is stretched
 *    And these global GS flags
 *        identityTransformation        1 == no need to stretch in GetCVT, etc.
 */
FS_VOID fsg_InitInterpreterTrans(fsg_SplineKey *key  )
{
    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);
    transMatrix                         *trans      = &key->currentTMatrix;

    key->identityTransformation = fsg_Identity( trans );
    if ( !key->identityTransformation )
    {
        fsg_GetMatrixStretch(&globalGS->xStretch, &globalGS->yStretch, trans); /*<8>*/
    }
    else
    {
        globalGS->xStretch = FIXED_ONE;
        globalGS->yStretch = FIXED_ONE;
    }

    globalGS->identityTransformation = key->identityTransformation;
    globalGS->non90DegreeTransformation = (FS_BYTE)fsg_Non90Degrees( trans );

    /* Use bit 1 of non90degreeTransformation to signify stretching.  stretch = 2 */
    if ( trans->transform[0][0] == trans->transform[1][1] || trans->transform[0][0] == -trans->transform[1][1] )
        globalGS->non90DegreeTransformation &= ~STRETCH;
    else
        globalGS->non90DegreeTransformation |= STRETCH;

#ifdef FS_HINTS
    globalGS->rtgah_data.fontflags = 0;
    globalGS->rtgah_data.stateflags = 0;
#endif

}


/**********************************************************************************/
/*
 * fsg_ScaleCVT
 *
 * The source Control Value Table in srcCVT is scaled and placed in cvt, an
 * array of F26DOT6. srcCVT is an array of SHORTs.
 * Added SWAPW and SWAPWPINC to take care of byte order bug -- mby 11/15/91
 */

static FS_VOID fsg_ScaleCVT(fsg_SplineKey *key, FS_LONG numCVT, F26DOT6 *cvt, FS_SHORT *srcCVT )
{
    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);
    scale_rec *rec = &globalGS->scaleCVT;
    FS_LONG V;
    FS_LONG N = rec->N;
    FS_LONG D = rec->D;
    FS_LONG Dover2 = D >> 1;
    int shift = rec->shift;
    FS_FIXED scl = rec->scl;

    if (globalGS->scaleFuncCVT == SFI_fnt_FRound)
    {
        while (numCVT--)
        {
            V = (FS_SHORT) SWAPWINC(srcCVT);
            FROUND(V, N, Dover2, shift);
            *cvt++ = V;
        }
    }
    else if (globalGS->scaleFuncCVT == SFI_fnt_SRound)
    {
        while (numCVT--)
        {
            V = (FS_SHORT) SWAPWINC(srcCVT);
            SROUND(V, N, D, Dover2);
            *cvt++ = V;
        }
    }
    else
    {
        while (numCVT--)
        {
            V = (FS_SHORT)SWAPWINC(srcCVT);
            *cvt++ = FixMul(scl, V);
        }
    }
}

/**********************************************************************************/
/*
 *    fsg_SetUpElement
 */
FS_VOID fsg_SetUpElement(_DS_ fsg_SplineKey *key, FS_LONG n)
{
    FS_BYTE             *workSpacePtr;
    fnt_ElementType     *elementPtr;
    fsg_OffsetInfo      *offsetPtr;
    TTF_MAXP *maxp = (TTF_MAXP *)(key->maxp);

    workSpacePtr        = (FS_BYTE *)(STATE.server->workspace);
    offsetPtr           = &(key->elementInfoRec.offsets[n]);
    elementPtr          = &(key->elementInfoRec.interpreterElements[n]);


    elementPtr->x        = (F26DOT6 *)(workSpacePtr + offsetPtr->newXOffset);
    elementPtr->y        = (F26DOT6 *)(workSpacePtr + offsetPtr->newYOffset);
    elementPtr->ox       = (F26DOT6 *)(workSpacePtr + offsetPtr->scaledXOffset);
    elementPtr->oy       = (F26DOT6 *)(workSpacePtr + offsetPtr->scaledYOffset);
    elementPtr->oox      = (F26DOT6 *)(workSpacePtr + offsetPtr->oldXOffset);
    elementPtr->ooy      = (F26DOT6 *)(workSpacePtr + offsetPtr->oldYOffset);
    elementPtr->sp       = (FS_USHORT *)(workSpacePtr + offsetPtr->startPointOffset);
    elementPtr->ep       = (FS_USHORT *)(workSpacePtr + offsetPtr->endPointOffset);
    elementPtr->onCurve  = (FS_BYTE *)(workSpacePtr + offsetPtr->onCurveOffset);
    elementPtr->f        = (FS_BYTE *)(workSpacePtr + offsetPtr->interpreterFlagsOffset);
    elementPtr->nc = 0;
    if ( n == TWILIGHTZONE )
    {
        /* int i, j; */
        elementPtr->sp[0] = 0;
        elementPtr->ep[0] = maxp->maxTwilightPoints - 1;
        elementPtr->nc    = MAX_TWILIGHT_CONTOURS;
    }
    else
    {
        fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);

#ifdef FS_EDGE_HINTS
        globalGS->maz_data.contours = (MAZOutlineContour *)(workSpacePtr + offsetPtr->MAZOutlineContoursOffset);
        globalGS->maz_data.points   = (MAZOutlinePoint *)  (workSpacePtr + offsetPtr->MAZOutlinePointsOffset);
        globalGS->maz_data.hSegs    = (MAZOutlineSegment *)(workSpacePtr + offsetPtr->MAZHSegsOffset);
        globalGS->maz_data.vSegs    = (MAZOutlineSegment *)(workSpacePtr + offsetPtr->MAZVSegsOffset);
        globalGS->maz_data.contourList = (MAZOutlineContour **)(workSpacePtr + offsetPtr->MAZOutlineContourListOffset);
        globalGS->maz_data.segPairs = (MAZOutlineSegPair *)(workSpacePtr + offsetPtr->MAZOutlineSegPairsOffset);
#endif
#ifdef FS_HINTS
        globalGS->rtgah_data.RTGAHPieces = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHPiecesOffset);
        globalGS->rtgah_data.RTGAHStrokes = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHStrokesOffset);
        globalGS->rtgah_data.RTGAHStrokesPlist = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHStrokesPlistOffset);
        globalGS->rtgah_data.RTGAHEdges = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHEdgesOffset);

        globalGS->rtgah_data.RTGAHXmin = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHXminOffset);
        globalGS->rtgah_data.RTGAHYmin = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHYminOffset);
        globalGS->rtgah_data.RTGAHXmax = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHXmaxOffset);
        globalGS->rtgah_data.RTGAHYmax = (FS_LONG *)(workSpacePtr + offsetPtr->RTGAHYmaxOffset);
#else
        globalGS = globalGS; /* avoid compiler warning */
#endif
    }
}


/**********************************************************************************/
/*
 *    fsg_IncrementElement
 */
FS_VOID fsg_IncrementElement(fsg_SplineKey *key, FS_LONG n, FS_LONG numPoints, FS_LONG numContours)
{
    fnt_ElementType     *elementPtr;

    elementPtr            = &(key->elementInfoRec.interpreterElements[n]);


    elementPtr->x        += numPoints;
    elementPtr->y        += numPoints;
    elementPtr->ox       += numPoints;
    elementPtr->oy       += numPoints;
    elementPtr->oox      += numPoints;
    elementPtr->ooy      += numPoints;
    elementPtr->sp       += numContours;
    elementPtr->ep       += numContours;
    elementPtr->onCurve  += numPoints;
    elementPtr->f        += numPoints;
    elementPtr->nc = 0;
}


/**********************************************************************************/
/*
 * fsg_ZeroOutTwilightZone            <3>
 */
static FS_VOID fsg_ZeroOutTwilightZone(fsg_SplineKey *key)
{
    FS_ULONG n = key->maxp->maxTwilightPoints;
    fnt_ElementType *elementPtr = &(key->elementInfoRec.interpreterElements[TWILIGHTZONE]);

    n *= sizeof(F26DOT6);
    SYS_MEMSET(elementPtr->x, 0, n);
    SYS_MEMSET(elementPtr->y, 0, n);
    SYS_MEMSET(elementPtr->ox, 0, n);
    SYS_MEMSET(elementPtr->oy, 0, n);
}

/**********************************************************************************/
/*
 *    Assign pgmList[] for each pre program
 */
static FS_VOID fsg_SetUpProgramPtrs(fsg_SplineKey* key, fnt_GlobalGraphicStateType* globalGS)
{
    if (key->read_programs)
    {
        TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);
        globalGS->pgmList[PREPROGRAM] = ttf->prep;
        globalGS->pgmList[FONTPROGRAM] = ttf->fpgm;
        globalGS->maxp = key->maxp;
        globalGS->cvtCount = (FS_USHORT)key->cvtCount;
        key->read_programs = 0;
    }
}


/**********************************************************************************/
static FS_VOID fsg_SetUpTablePtrs(_DS_ fsg_SplineKey* key)
{
    FS_BYTE  *private_FontSpacePtr;
    FS_BYTE  **memoryBases = key->memoryBases;
    fnt_GlobalGraphicStateType* globalGS = GLOBALGSTATE(key);
    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);

    private_FontSpacePtr = (FS_BYTE *)(memoryBases[PRIVATE_FONT_SPACE_BASE]);
    switch (globalGS->pgmIndex)
    {
    case PREPROGRAM:
        globalGS->controlValueTable    = (F26DOT6*)(private_FontSpacePtr + key->offset_controlValues);
        /* FALLTHROUGH */
    case FONTPROGRAM:
#if SIZEOF_LONG==8    /* on Alpha */
        {
            FS_INT64 dotptr;

            dotptr = (FS_INT64)(private_FontSpacePtr + key->offset_storage);
            dotptr = ((dotptr + 7) / 8) * 8; /* boundary alignment on Alpha */
            globalGS->store = (F26DOT6*) dotptr;
        }
#else
        globalGS->store    = (F26DOT6*)(private_FontSpacePtr + key->offset_storage);
#endif
        globalGS->funcDef    = (fnt_funcDef *)(ttf->functionDefs);
        globalGS->instrDef    = (fnt_instrDef *)(ttf->instructionDefs);
        globalGS->stackBase =
            (F26DOT6*)(STATE.server->workspace +
                       key->elementInfoRec.stackBaseOffset);
        globalGS->function  =
            (FntFunc*)memoryBases[VOID_FUNC_PTR_BASE];
        break;
    default:
        break;
    }
}
/**********************************************************************************/
/*
 * fsg_RunPreProgram
 *
 * Runs the pre-program and scales the control value table
 *
 */
FS_LONG fsg_RunPreProgram(_DS_ fsg_SplineKey *key)
{
    FS_LONG result;
    FS_SHORT *cvtSrc;
    F26DOT6* cvt = (F26DOT6*)(key->memoryBases[PRIVATE_FONT_SPACE_BASE] + key->offset_controlValues);
    TTF *ttf = (TTF *)(key->sfnt->lfnt->fnt);

    fnt_GlobalGraphicStateType    *globalGS = GLOBALGSTATE(key);
    FS_LONG numCvt;
    cvtSrc = (FS_SHORT *)(ttf->cvt);
    numCvt = ttf->num_cvt;

    /* Set up the engine compensation array for the interpreter */
    /* This will be indexed into by the booleans in some instructions */
    globalGS->engine[0] = globalGS->engine[3] = 0;                            /* Gray and ? distance */
    globalGS->engine[1] = FIXEDTODOT6(FIXEDSQRT2 - key->pixelDiameter);       /* Black distance */
    globalGS->engine[2] = -globalGS->engine[1];                               /* White distance */

    globalGS->init           = 1;
    globalGS->pixelsPerEm    = FS_ROUND(key->interpScalar);
    globalGS->pointSize      = FS_ROUND(key->fixedPointSize);
    globalGS->fpem           = key->interpScalar;
    result = fsg_SetDefaults(key );
    if (result)
        return result;        /* Set graphic state to default values */
    globalGS->localParBlock = globalGS->defaultParBlock;    /* copy gState parameters */
    key->tInfo.x = globalGS->xStretch;
    key->tInfo.y = globalGS->yStretch;

    fsg_ScaleCVT(key, numCvt, cvt, cvtSrc);

    globalGS->pgmIndex = PREPROGRAM;
    fsg_SetUpProgramPtrs(key, globalGS);

    /** TWILIGHT ZONE ELEMENT **/
    fsg_SetUpElement(_PS_ key, (FS_LONG)TWILIGHTZONE);
    fsg_SetUpElement(_PS_ key, (FS_LONG)GLYPHELEMENT);    /* Oak */
    fsg_ZeroOutTwilightZone(key);

    fsg_SetUpTablePtrs(_PS_ key);
    globalGS->glyphProgram = 0;


#ifdef FS_HINTS
#ifdef EDIT_MODE
    result = fnt_Execute(key->elementInfoRec.interpreterElements,
                         preProgram,
                         preProgram + preProgramLength,
                         globalGS);
#else
    result = fnt_Execute(key->elementInfoRec.interpreterElements,
                         (FS_BYTE *)(globalGS->pgmList[PREPROGRAM]),
                         (FS_BYTE *)(globalGS->pgmList[PREPROGRAM]) + ttf->num_prep,
                         globalGS);
#endif /* EDIT_MODE */
#endif /* FS_HINTS */

    key->executePrePgm = 0;
    if ( !(globalGS->localParBlock.instructControl & DEFAULTFLAG) )
        globalGS->defaultParBlock = globalGS->localParBlock;    /* change default parameters */
    return result;
}


/**********************************************************************************/
/* Set default values for all variables in globalGraphicsState DefaultParameterBlock
 *    Eventually, we should provide for a Default preprogram that could optionally be
 *    run at this time to provide a different set of default values.
 */
FS_LONG fsg_SetDefaults(fsg_SplineKey* key )
{
    fnt_GlobalGraphicStateType *globalGS = GLOBALGSTATE(key);
    fnt_ParameterBlock *par = &globalGS->defaultParBlock;

    par->RoundValue  = RFI_fnt_RoundToGrid;
    par->minimumDistance = fnt_pixelSize;
    par->wTCI = fnt_pixelSize * 17 / 16;
    par->sWCI = 0;
    par->sW   = 0;
    par->scaledSW = 0;
    par->autoFlip = 1;
    par->deltaBase = 9;
    par->deltaShift = 3;
    par->angleWeight = 128;
    par->scanControl = 0;
    par->instructControl = 0;
    par->pad = 0;
    par->period = 0;
    par->period45 = 0;
    par->periodMask = 0;
    par->phase = 0;
    par->threshold = 0;
    return 0;
}



/****************************************************************/
#define GET_SHORT(p) (FS_SHORT)(((FS_USHORT)p[0] << 8)|p[1]); p+=2

/****************************************************************/
static int sfnt_UnfoldCurve(fsg_SplineKey* key,
                     sfnt_PackedSplineFormat* charData,
                     FS_LONG* sizeOfInstructions,
                     FS_BYTE** instructionPtr,
                     FS_LONG length)
{
    int z;
    FS_BYTE flag, *byteP, *byteP2;
    FS_BYTE *p;
    F26DOT6* longP;
    fnt_ElementType *elementPtr;
    int numPoints;
#ifdef FS_STIK
    int is_outline;
#endif

    /*FS_PRINTF(("sfnt_UnfoldCurve:\n")); */

    elementPtr = &(key->elementInfoRec.interpreterElements[GLYPHELEMENT]);

    if ( length <= 0 || charData->numberContours == 0)
    {
        elementPtr->nc = 0;

        elementPtr->sp[0] = 0;
        elementPtr->ep[0] = 0;

        *instructionPtr     = 0;
        *sizeOfInstructions = 0;

        elementPtr->onCurve[0] = ONCURVE;
        elementPtr->oox[0] = 0;
        elementPtr->ooy[0] = 0;

        return NO_ERR; /***************** exit here ! *****************/
    }

    elementPtr->nc = charData->numberContours;
    z = (int)elementPtr->nc;

    if ( z < 0 || z > key->maxp->maxContours )
    {
#ifdef FS_DEBUG
        FS_PRINTF(("contours=%d maxp->maxContours=%d\n", z, key->maxp->maxContours));
#endif
        return CONTOUR_DATA_ERR;
    }

    {
        /* <4> */
        FS_USHORT* sp = elementPtr->sp;
        FS_USHORT* ep = elementPtr->ep;
        FS_SHORT* charDataEP = charData->endPoints;
        FS_LONG i;
        *sp++ = 0;
        *ep = (FS_SHORT)SWAPWINC(charDataEP);
        i = z - 1;
        while (i--)
        {
            *sp++ = *ep++ + 1;
            *ep = (FS_SHORT)SWAPWINC(charDataEP);
        }
    }

    numPoints = elementPtr->ep[elementPtr->nc - 1] + 1;
    if (numPoints < 0 || numPoints > key->maxp->maxPoints )
        return POINTS_DATA_ERR;

    *sizeOfInstructions = charData->numberInstructions;
    *instructionPtr = charData->instructions;

    /* Do flags */
    p = (FS_BYTE*)charData->flags;

#ifdef FS_STIK
    is_outline = charData->flags[0] & OUTLINE_CHAR;
#endif
    byteP = elementPtr->onCurve;
    byteP2 = byteP + numPoints;            /* only need to set this guy once */
    while (byteP < byteP2)
    {
        *byteP++ = flag = *p++;
        if ( flag & REPEAT_FLAGS )
        {
            FS_LONG count = *p++;
            while (count--)
                *byteP++ = flag;
        }
    }

    /* Do X first */
    z = 0;
    byteP = elementPtr->onCurve;
    longP = elementPtr->oox;
    /* FS_PRINTF(("x\n")); */
    while (byteP < byteP2)
    {
        if ( (flag = *byteP++) & XSHORT )
        {
            if ( flag & SHORT_X_IS_POS )
                z += *p++;
            else
                z -= *p++;
        }
        else if ( !(flag & NEXT_X_IS_ZERO) )
        {
            z += GET_SHORT(p);
        }
        /* FS_PRINTF(("\t%d  %d\n",z,flag & ONCURVE));*/
        *longP++ = (F26DOT6)z;
    }

    /* Now Do Y */
    z = 0;
    byteP = elementPtr->onCurve;
    longP = elementPtr->ooy;
    /* FS_PRINTF(("y\n")); */
    while (byteP < byteP2)
    {
        if ( (flag = *byteP) & YSHORT )
        {
            if ( flag & SHORT_Y_IS_POS )
                z += *p++;
            else
                z -= *p++;
        }
        else if ( !(flag & NEXT_Y_IS_ZERO) )
        {
            z += GET_SHORT(p);
        }
        /* FS_PRINTF(("\t%d %d\n",z,flag & ONCURVE)); */
        *longP++ = (F26DOT6)z;
        *byteP++ = flag & (FS_BYTE)ONCURVE; /* Filter out unwanted stuff */

    }

#ifdef FS_STIK
    if (is_outline)
        elementPtr->onCurve[0] |= OUTLINE_CHAR;
#endif

#ifdef NC_DEBUG
    {
        FS_LONG *x, *y, *ox, *oy, *oox, *ooy;
        FS_BYTE *f, *oc;
        int i;

        f = elementPtr->f;
        oc = elementPtr->onCurve;
        x = elementPtr->x;
        y = elementPtr->y;
        ox = elementPtr->ox;
        oy = elementPtr->oy;
        oox = elementPtr->oox;
        ooy = elementPtr->ooy;
        for (i = 0; i < numPoints; i++)
        {
            *f++ = 0;
            oc++;
            *x++ = 0;
            *y++ = 0;
            *ox++ = 0;
            *oy++ = 0;
            oox++;
            ooy++;
        }

        for (/**/; i < (numPoints + PHANTOMCOUNT); i++)
        {
            *f++ = 0;
            *oc++ = 0;
            *x++ = 0;
            *y++ = 0;
            *ox++ = 0;
            *oy++ = 0;
            *oox++ = 0;
            *ooy++ = 0;
        }
    }
#endif
    return NO_ERR;
}

/****************************************************************/
int get_dropout_control(fsg_SplineKey *key)
{
    FS_LONG scanControl = key->scanControl;
    FS_SHORT threshold = (FS_SHORT)(scanControl & 0xFF);
    FS_SHORT ppm = (FS_SHORT)(key->metricScalar >> 16);
    FS_FIXED *m = (FS_FIXED *) &key->currentTMatrix.transform[0][0];
    FS_SHORT is_rotated = m[1] && m[3];
    FS_SHORT is_stretched = m[0] != m[4];


    /* hints in PREP or glyph turned DropoutControl OFF */
    if (scanControl & NODOCONTROL)
        return 0;

    if (threshold == 0xFF)
        threshold = 32767;

    if ((scanControl & 0x100) && (ppm <= threshold))
        return 1;

    if ((scanControl & 0x200) && is_rotated)
        return 1;

    if ((scanControl & 0x400) && is_stretched)
        return 1;

    if ((scanControl & 0x800) && (ppm > threshold))
        return 0;

    if ((scanControl & 0x1000) && (!is_rotated))
        return 0;

    if ((scanControl & 0x2000) && (!is_stretched))
        return 0;

    return 0;

}
/**********************************************************************************/
/*
 *    Return 45 degreeness
 */
static FS_BYTE fsg_Max45Trick(FS_FIXED x, FS_FIXED y, FS_FIXED* stretch)
{
    if (x < 0) x = -x;
    if (y < 0) y = -y;
    if (x < y)        /* make sure x > y */
    {
        x = y;
    }

    *stretch = x;
    return 0;
}

/**********************************************************************************/
/*
 *    Sets [xy]Stretch factors to be applied before hinting.
 * <8> don't return need for point reversal
 */
static FS_VOID fsg_GetMatrixStretch(FS_FIXED* xStretch,  FS_FIXED* yStretch, transMatrix* trans)
{
    FS_FIXED* matrix = &trans->transform[0][0];
    FS_FIXED x, y;

    x = *matrix++;
    y = *matrix++;
    fsg_Max45Trick(x, y, xStretch);

    matrix++;

    x = *matrix++;
    y = *matrix;
    fsg_Max45Trick(x, y, yStretch);
}
/**********************************************************************************/
/*
 *    Call this guy before you use the matrix.  He does two things:
 *        He folds any perspective-translation back into perspective,
 *         and then changes the [2][2] element from a FRACT to a FS_FIXED.
 *        He then Finds the largest scaling value in the matrix and
 *         removes it from then and folds it into metricScalar.
 */
FS_VOID fsg_ReduceMatrix(fsg_SplineKey* key)
{
    FS_FIXED a;
    FS_FIXED* matrix = &key->currentTMatrix.transform[0][0];
    FRACT bottom = matrix[8];
    /*
     *    First, fold translation into perspective, if any.
     */
    if (0 != (a = matrix[2]))
    {
        matrix[0] -= LongMulDiv(a, matrix[6], bottom);
        matrix[1] -= LongMulDiv(a, matrix[7], bottom);
    }
    if (0 != (a = matrix[5]))
    {
        matrix[3] -= LongMulDiv(a, matrix[6], bottom);
        matrix[4] -= LongMulDiv(a, matrix[7], bottom);
    }
    matrix[6] = matrix[7] = 0;
    matrix[8] = FRACT2FIX(bottom);        /* make this guy a FS_FIXED for XYMul routines */

    /*
     *    Now suck out the biggest scaling factor: matrix[0] [1] [3] [4]
     *    Should be folded into GetMatrixStretch, when I understand xformed-components <4>
     */

    /* the following line confuses the ALPHA compiler
     * a = fsg_MaxAbs( *matrix++, *matrix++ ); matrix++;
     */
    a = fsg_MaxAbs( matrix[0], matrix[1]);
    matrix += 3;
    a = fsg_MaxAbs( a, *matrix++ );
    a = fsg_MaxAbs( a, *matrix );

    if (a != FIXED_ONE)
    {
        *matrix = FixDiv(*matrix, a);
        --matrix;
        *matrix = FixDiv(*matrix, a);
        matrix -= 2;
        *matrix = FixDiv(*matrix, a);
        --matrix;
        *matrix = FixDiv(*matrix, a);

        key->metricScalar = FixMul( key->metricScalar, a );
    }
    /*    Now the matrix is smaller and metricScalar is bigger */
}

/**********************************************************************************/
#else
FS_VOID delete_key(_DS_ FS_VOID *p)
{
    FS_state_ptr = FS_state_ptr;
    p = p;
}
#endif /* ifdef FS_RENDER*/
